<?php
// public/driver/resubmit.php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$today = (new DateTime('today'))->format('Y-m-d');
$ok=''; $err='';

// Pull partner id for path hint and reuse
$driverRow = $conn->prepare("SELECT partner_id FROM drivers WHERE id=?");
$driverRow->bind_param('i', $driver_id);
$driverRow->execute();
$driverInfo = $driverRow->get_result()->fetch_assoc();
$partner_id = $driverInfo['partner_id'] ?? ('DRV'.$driver_id);
$driverRow->close();

// Load submission by id or by date
$submission = null;
if (isset($_GET['id']) && ctype_digit($_GET['id'])) {
  $sid = (int)$_GET['id'];
  $st = $conn->prepare("SELECT * FROM submissions WHERE id=? AND driver_id=? LIMIT 1");
  $st->bind_param('ii', $sid, $driver_id);
  $st->execute();
  $submission = $st->get_result()->fetch_assoc();
} elseif (!empty($_GET['date'])) {
  $d = $_GET['date'];
  $st = $conn->prepare("SELECT * FROM submissions WHERE driver_id=? AND service_date=? LIMIT 1");
  $st->bind_param('is', $driver_id, $d);
  $st->execute();
  $submission = $st->get_result()->fetch_assoc();
}

// If no submission yet for that day, allow creating one quickly
$allow_create = false;
if (!$submission && !empty($_GET['date'])) {
  $d = $_GET['date'];
  // must be past date and no submission must exist for it
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $d) && $d < $today) {
    $chk = $conn->prepare("SELECT id FROM submissions WHERE driver_id=? AND service_date=? LIMIT 1");
    $chk->bind_param('is', $driver_id, $d);
    $chk->execute();
    if (!$chk->get_result()->fetch_assoc()) {
      $allow_create = true;
    }
  } // if a submission was found by date, $submission will be populated and this block is skipped.
}

if (!$submission && !$allow_create) {
  http_response_code(404);
  exit('Invalid submission');
}

// If exists but approved, block edits
if ($submission && strtolower($submission['status']) === 'approved') {
  http_response_code(403);
  exit('Already approved, cannot edit.');
}

// Handle POST
if (is_valid_post()) {
  $service_date = $submission ? $submission['service_date'] : (trim($_POST['service_date'] ?? ''));
  $orders = trim($_POST['orders_count'] ?? '');
  $cash   = trim($_POST['cash_deposited'] ?? '');
  $trx    = trim($_POST['trx_no'] ?? '');
  $remarks= trim($_POST['remarks'] ?? '');

  // Date checks
  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $service_date)) {
    $err = 'Invalid date.';
  } elseif ($service_date >= $today) {
    $err = 'You cannot submit for today or future dates.';
  } elseif ($orders === '' || !ctype_digit($orders)) {
    $err = 'Orders must be a whole number.';
  } elseif ($cash === '' || !preg_match('/^\d+(\.\d{1,3})?$/', $cash)) {
    $err = 'Cash must be a number with up to 3 decimals.';
  }

  // Optional files save helper - saves to /storage/submissions/
  function save_upload_optional(string $field, string $partner_id, string $service_date, string $type): ?string {
    if (!isset($_FILES[$field]) || $_FILES[$field]['error'] !== UPLOAD_ERR_OK) return null;
    $ext = strtolower(pathinfo($_FILES[$field]['name'], PATHINFO_EXTENSION));
    if ($ext === '') $ext = 'jpg';

    // Build path: /storage/submissions/PARTNER_ID/YYYY-MM/
    $rel_path = $partner_id . '/' . substr($service_date, 0, 7);
    $abs_dir = STORAGE_DIR . '/submissions/' . $rel_path;
    @mkdir($abs_dir, 0775, true);

    $filename = $type . '_' . uniqid() . '.' . $ext;
    $abs_path = $abs_dir . '/' . $filename;

    if (!move_uploaded_file($_FILES[$field]['tmp_name'], $abs_path)) return null;

    // Return storage-relative path for DB
    return 'storage/submissions/' . $rel_path . '/' . $filename;
  }

  if (!$err) {
    // If no prior submission and allow_create, insert new
    if (!$submission && $allow_create) {
      // Ensure no existing one sneaked in
      $chk = $conn->prepare("SELECT id FROM submissions WHERE driver_id=? AND service_date=? LIMIT 1");
      $chk->bind_param('is', $driver_id, $service_date);
      $chk->execute();
      if ($chk->get_result()->fetch_assoc()) {
        $err = 'A submission already exists for that date.';
      } else {
        $ss_orders    = save_upload_optional('ss_orders', $partner_id, $service_date, 'orders');
        $ss_deposit_1 = save_upload_optional('ss_deposit_1', $partner_id, $service_date, 'deposit1');
        $ss_deposit_2 = save_upload_optional('ss_deposit_2', $partner_id, $service_date, 'deposit2');

        $ins = $conn->prepare("
          INSERT INTO submissions
            (driver_id, service_date, orders_count, cash_deposited, ss_orders, ss_deposit_1, ss_deposit_2, trx_no, remarks, status, created_at, updated_at)
          VALUES (?,?,?,?,?,?,?,?,?,'pending',NOW(),NOW())
        ");
        $ins->bind_param(
          'isidsssss',
          $driver_id, $service_date, $orders, $cash, $ss_orders, $ss_deposit_1, $ss_deposit_2, $trx, $remarks
        );
        if ($ins->execute()) {
          $ok = 'Submission created and sent for approval.';
          // reload submission to keep the page consistent
          $sid = $conn->insert_id;
          $re = $conn->prepare("SELECT * FROM submissions WHERE id=? AND driver_id=?");
          $re->bind_param('ii', $sid, $driver_id);
          $re->execute();
          $submission = $re->get_result()->fetch_assoc();
        } else {
          $err = 'Create failed.';
        }
      }
    } else {
      // Update existing, only pending or rejected
      $sid = (int)$submission['id'];

      $new_ss_orders    = save_upload_optional('ss_orders', $partner_id, $service_date, 'orders');
      $new_ss_deposit_1 = save_upload_optional('ss_deposit_1', $partner_id, $service_date, 'deposit1');
      $new_ss_deposit_2 = save_upload_optional('ss_deposit_2', $partner_id, $service_date, 'deposit2');

      // Build dynamic update to keep old paths if files not uploaded
      $sql = "UPDATE submissions SET orders_count=?, cash_deposited=?, trx_no=?, remarks=?, status='pending', updated_at=NOW()";
      $types = 'idss';
      $args  = [ (int)$orders, (float)$cash, $trx, $remarks ];

      if ($new_ss_orders !== null)    { $sql .= ", ss_orders=?";    $types .= 's'; $args[] = $new_ss_orders; }
      if ($new_ss_deposit_1 !== null) { $sql .= ", ss_deposit_1=?"; $types .= 's'; $args[] = $new_ss_deposit_1; }
      if ($new_ss_deposit_2 !== null) { $sql .= ", ss_deposit_2=?"; $types .= 's'; $args[] = $new_ss_deposit_2; }

      $sql .= " WHERE id=? AND driver_id=? AND service_date=?";
      $types .= 'iis';
      $args[] = $sid; $args[] = $driver_id; $args[] = $service_date;

      $up = $conn->prepare($sql);
      $up->bind_param($types, ...$args);
      if ($up->execute()) {
        $ok = 'Submission updated and re-sent for approval.';
        // refresh submission
        $re = $conn->prepare("SELECT * FROM submissions WHERE id=? AND driver_id=?");
        $re->bind_param('ii', $sid, $driver_id);
        $re->execute();
        $submission = $re->get_result()->fetch_assoc();
      } else {
        $err = 'Update failed.';
      }
    }
  }
}

// UI start
if (is_file(__DIR__.'/../../includes/driver_ui_start.php')) {
  include __DIR__.'/../../includes/driver_ui_start.php';
} else {
  echo '<!doctype html><html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">';
  echo '<title>Resubmit</title></head><body class="bg-light"><div class="container py-3">';
}
?>
<?php
function resubmit_badge($s){
  $s = strtolower((string)$s);
  if ($s==='rejected') return '<span class="badge text-bg-danger">Rejected</span>';
  return '<span class="badge text-bg-warning text-dark">Pending</span>';
}
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h5 class="mb-0"><?php echo $submission ? 'Review or update submission' : 'Create a submission'; ?></h5>
  <a class="btn btn-outline-secondary btn-sm" href="<?php echo base_url('driver/dashboard.php'); ?>">Back</a>
</div>

<?php if ($ok): ?><div class="alert alert-success py-2"><?php echo e($ok); ?></div><?php endif; ?>
<?php if ($err): ?><div class="alert alert-danger py-2"><?php echo e($err); ?></div><?php endif; ?>

<form method="post" enctype="multipart/form-data">
  <input type="hidden" name="csrf" value="<?php echo e(get_csrf_token()); ?>">

  <!-- Main details -->
  <div class="card mb-3">
    <div class="card-body">
      <div class="row g-3">
        <div class="col-12 col-md-6">
          <label class="form-label">Date</label>
          <input class="form-control bg-light" name="service_date"
                 value="<?php echo e($submission['service_date'] ?? ($_GET['date'] ?? '')); ?>"
                 <?php echo ($submission || !empty($_GET['date'])) ? 'readonly' : 'required'; ?>>
          <div class="form-text">Submissions are for past dates only.</div>
        </div>

        <?php if ($submission): ?>
        <div class="col-12 col-md-6">
          <label class="form-label">Status</label>
          <div class="form-control-plaintext"><?php echo resubmit_badge($submission['status']); ?></div>
        </div>
        <?php endif; ?>

        <div class="col-12 col-md-6">
          <label class="form-label">Total orders</label>
          <input class="form-control" name="orders_count" inputmode="numeric" pattern="\d+"
                 value="<?php echo e($submission['orders_count'] ?? ''); ?>" required>
        </div>

        <div class="col-12 col-md-6">
          <label class="form-label">Total cash deposited (OMR)</label>
          <input class="form-control" name="cash_deposited" inputmode="decimal" pattern="\d+(\.\d{1,3})?"
                 value="<?php echo e(isset($submission['cash_deposited']) ? number_format((float)$submission['cash_deposited'], 3, '.', '') : ''); ?>" required>
          <div class="form-text">E.g. 123.450</div>
        </div>
      </div>
    </div>
  </div>

  <!-- Attachments -->
  <div class="card mb-3">
    <div class="card-header">Attachments</div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-12 col-md-4">
          <label class="form-label">Order screenshot</label>
          <input type="file" name="ss_orders" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf">
          <?php if (!empty($submission['ss_orders'])): ?>
            <div class="form-text small"><a href="#" data-bs-toggle="modal" data-bs-target="#fileViewerModal" data-url="<?php echo e(file_url($submission['ss_orders'])); ?>">View current file</a></div>
          <?php endif; ?>
        </div>
        <div class="col-12 col-md-4">
          <label class="form-label">Deposit slip 1</label>
          <input type="file" name="ss_deposit_1" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf">
          <?php if (!empty($submission['ss_deposit_1'])): ?>
            <div class="form-text small"><a href="#" data-bs-toggle="modal" data-bs-target="#fileViewerModal" data-url="<?php echo e(file_url($submission['ss_deposit_1'])); ?>">View current file</a></div>
          <?php endif; ?>
        </div>
        <div class="col-12 col-md-4">
          <label class="form-label">Deposit slip 2 (Optional)</label>
          <input type="file" name="ss_deposit_2" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf">
          <?php if (!empty($submission['ss_deposit_2'])): ?>
            <div class="form-text small"><a href="#" data-bs-toggle="modal" data-bs-target="#fileViewerModal" data-url="<?php echo e(file_url($submission['ss_deposit_2'])); ?>">View current file</a></div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- Optional Info -->
  <div class="card mb-3">
    <div class="card-header">
      <a href="#optional-info" data-bs-toggle="collapse" class="text-decoration-none text-dark">
        Optional Information
      </a>
    </div>
    <div class="collapse show" id="optional-info">
      <div class="card-body">
        <div class="row g-3">
          <div class="col-12 col-md-6">
            <label class="form-label">Bank TRX No.</label>
            <input class="form-control" name="trx_no" value="<?php echo e($submission['trx_no'] ?? ''); ?>">
          </div>
          <div class="col-12">
            <label class="form-label">Remarks</label>
            <textarea class="form-control" name="remarks" rows="2"><?php echo e($submission['remarks'] ?? ''); ?></textarea>
          </div>
        </div>
      </div>
    </div>
  </div>

  <div class="d-grid">
    <button class="btn btn-danger btn-lg"><?php echo $submission ? 'Update Submission' : 'Create Submission'; ?></button>
  </div>
</form>

<!-- File Viewer Modal -->
<div class="modal fade" id="fileViewerModal" tabindex="-1" aria-labelledby="fileViewerModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="fileViewerModalLabel">File Preview</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body text-center">
        <!-- Content will be injected here by JS -->
      </div>
    </div>
  </div>
</div>

<?php
if (is_file(__DIR__.'/../../includes/driver_ui_end.php')) {
  include __DIR__.'/../../includes/driver_ui_end.php';
} else {
  echo '</div><script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script></body></html>';
}
?>
<script>
document.addEventListener('DOMContentLoaded', function () {
  const fileViewerModal = document.getElementById('fileViewerModal');
  if (fileViewerModal) {
    fileViewerModal.addEventListener('show.bs.modal', function (event) {
      const button = event.relatedTarget;
      const url = button.getAttribute('data-url');
      const modalBody = fileViewerModal.querySelector('.modal-body');
      const isImage = /\.(jpe?g|png|gif|webp)$/i.test(url);

      if (isImage) {
        modalBody.innerHTML = `<img src="${url}" class="img-fluid" alt="File preview">`;
      } else {
        modalBody.innerHTML = `<iframe src="${url}" style="width:100%; height:75vh;" frameborder="0"></iframe>`;
      }
    });
  }
});
</script>
