<?php
// public/driver/payout_view.php
// Driver payout details view, bound to logged in driver

require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$title = 'Payout details';

$cycle_id = (int)($_GET['cycle_id'] ?? 0);
if ($cycle_id <= 0) {
    die('Missing payout cycle.');
}

// Load cycle
$stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$cycle = $res->fetch_assoc();
$stmt->close();

if (!$cycle) {
    die('Payout cycle not found.');
}

// Load payout item for this driver and cycle
$sql = "
    SELECT
      pi.*,
      d.first_name,
      d.last_name
    FROM payout_items pi
    JOIN drivers d ON d.id = pi.driver_id
    WHERE pi.cycle_id = ? AND pi.driver_id = ?
    LIMIT 1
";
$stmt = $conn->prepare($sql);
$stmt->bind_param('ii', $cycle_id, $driver_id);
$stmt->execute();
$res  = $stmt->get_result();
$item = $res->fetch_assoc();
$stmt->close();

if (!$item) {
    die('No payout entry found for this cycle.');
}

// Load monthly earnings (FIX)
$earnSQL = "
    SELECT completed_deliveries, operations_fee, tips, incentives, compensations, deductions
    FROM driver_earnings_monthly
    WHERE driver_id = ? AND month = ?
    LIMIT 1
";
$stmtE = $conn->prepare($earnSQL);
$stmtE->bind_param('is', $driver_id, $cycle['cycle_month']);
$stmtE->execute();
$resE  = $stmtE->get_result();
$earn  = $resE->fetch_assoc();
$stmtE->close();

$completed_deliveries = (int)($earn['completed_deliveries'] ?? 0);
$operations_fee       = (float)($earn['operations_fee'] ?? 0);
$tips                 = (float)($earn['tips'] ?? 0);
$incentives           = (float)($earn['incentives'] ?? 0);
$compensations        = (float)($earn['compensations'] ?? 0);
$deductions           = (float)($earn['deductions'] ?? 0);

// Finance note
$finance_note = trim($item['finance_note'] ?? '');

// Load breakdown rows
$stmt = $conn->prepare("
    SELECT
      pb.*,
      da.remarks          AS adj_remarks,
      da.is_advance       AS adj_is_advance,
      da.is_shortage      AS adj_is_shortage,
      da.type             AS adj_type,
      ac.name             AS category_name
    FROM payout_breakdowns pb
    LEFT JOIN driver_adjustments da
      ON pb.reference_type = 'adjustment' AND pb.reference_id = da.id
    LEFT JOIN adjustment_categories ac
      ON da.category_id = ac.id
    WHERE pb.payout_item_id = ?
    ORDER BY pb.created_at ASC, pb.id ASC
");
$stmt->bind_param('i', $item['id']);
$stmt->execute();
$res   = $stmt->get_result();
$rows  = [];
while ($r = $res->fetch_assoc()) {
    $rows[] = $r;
}
$stmt->close();

// Driver name
$driver_name = trim(($item['first_name'] ?? '') . ' ' . ($item['last_name'] ?? ''));
if ($driver_name === '') {
    $driver_name = 'Driver';
}

// Totals
$total_earnings = (float)$item['total_earnings'];
$total_pos      = (float)$item['total_positive_adjustments'];
$total_neg      = (float)$item['total_negative_adjustments'];
$total_adv      = (float)$item['total_advances'];
$total_short    = (float)$item['total_shortages'];
$net_payout     = (float)$item['net_payout'];

$net_adjustments = $total_pos - $total_neg;

// Group rows
$earning_rows  = [];
$adjust_rows   = [];
$advance_rows  = [];
$shortage_rows = [];

foreach ($rows as $r) {
    $type = $r['type'];

    if ($type === 'earning') {
        $earning_rows[] = $r;
    } else {
        $is_adv  = (int)($r['adj_is_advance'] ?? 0) === 1;
        $is_shrt = (int)($r['adj_is_shortage'] ?? 0) === 1;

        if ($is_adv) {
            $advance_rows[] = $r;
        } elseif ($is_shrt) {
            $shortage_rows[] = $r;
        } else {
            $adjust_rows[] = $r;
        }
    }
}

require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<div class="mb-4" style="max-width: 640px; margin: 0 auto;">

  <style>
    .payout-total-card {
      background: #ffffff;
      border-radius: 14px;
      padding: 14px 16px;
      box-shadow: 0 1px 4px rgba(15, 23, 42, 0.1);
      margin-bottom: 12px;
    }
    .payout-total-label {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: .08em;
      color: #6b7280;
      margin-bottom: 4px;
    }
    .payout-total-value {
      font-size: 24px;
      font-weight: 700;
      color: #111827;
    }
    .payout-total-sub {
      font-size: 13px;
      color: #6b7280;
    }
    .mini-grid {
      display: grid;
      grid-template-columns: repeat(2, minmax(0, 1fr));
      gap: 8px;
      margin-bottom: 14px;
    }
    .mini-card {
      background: #ffffff;
      border-radius: 10px;
      padding: 8px 10px;
      box-shadow: 0 1px 3px rgba(15, 23, 42, 0.08);
    }
    .mini-label {
      font-size: 11px;
      text-transform: uppercase;
      letter-spacing: .06em;
      color: #6b7280;
      margin-bottom: 2px;
    }
    .mini-value {
      font-size: 15px;
      font-weight: 600;
    }
    .mini-hint {
      font-size: 11px;
      color: #9ca3af;
    }
    .section-card {
      background: #ffffff;
      border-radius: 12px;
      box-shadow: 0 1px 4px rgba(15, 23, 42, 0.08);
      margin-bottom: 10px;
      overflow: hidden;
    }
    .section-header {
      padding: 8px 12px;
      display: flex;
      justify-content: space-between;
      align-items: center;
      cursor: pointer;
    }
    .section-title {
      font-size: 14px;
      font-weight: 600;
    }
    .section-extra {
      font-size: 12px;
      color: #6b7280;
    }
    .section-body {
      padding: 0 12px 8px;
    }
    .line-row {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 5px 0;
      border-top: 1px solid #e5e7eb;
    }
    .line-label {
      font-size: 13px;
      color: #6b7280;
    }
    .line-amount {
      font-size: 13px;
      font-weight: 500;
    }
    .line-amount.pos { color: #16a34a; }
    .line-amount.neg { color: #dc2626; }

    .chevron { font-size: 12px; color: #6b7280; }

    .note-card {
      background: #fff7ed;
      border-radius: 8px;
      padding: 8px 10px;
      font-size: 12px;
      color: #92400e;
      border: 1px solid #fed7aa;
      margin-bottom: 12px;
    }
  </style>

  <div class="d-flex justify-content-between align-items-center mb-2">
    <a href="payouts_list.php" class="btn btn-sm btn-outline-secondary">Back to payouts</a>
    <a href="payout_slip.php?cycle_id=<?php echo $cycle_id; ?>" target="_blank" class="btn btn-sm btn-danger">Payout slip</a>
  </div>

  <div class="mb-2">
    <div class="small text-muted"><?php echo e($driver_name); ?></div>
    <h5 class="mb-1">Payout for <?php echo e($cycle['cycle_month']); ?></h5>
    <div class="small text-muted">
      <?php echo e($cycle['period_start']); ?> to <?php echo e($cycle['period_end']); ?>
    </div>
  </div>

  <div class="payout-total-card">
    <div class="payout-total-label">Final payout</div>
    <div class="payout-total-value">
      <?php echo number_format($net_payout, 3); ?> OMR
    </div>
    <div class="payout-total-sub">This is your total payout for this period.</div>
  </div>

  <?php if ($finance_note !== ''): ?>
    <div class="note-card">
      <strong>Finance note:</strong><br><?php echo nl2br(e($finance_note)); ?>
    </div>
  <?php endif; ?>

  <div class="mini-grid">
    <div class="mini-card">
      <div class="mini-label">Earnings</div>
      <div class="mini-value text-success"><?php echo number_format($total_earnings, 3); ?></div>
      <div class="mini-hint">Orders, tips and incentives</div>
    </div>

    <div class="mini-card">
      <div class="mini-label">Net adjustments</div>
      <div class="mini-value <?php echo $net_adjustments >= 0 ? 'text-success' : 'text-danger'; ?>">
        <?php echo number_format($net_adjustments, 3); ?>
      </div>
      <div class="mini-hint">Bonuses and penalties</div>
    </div>

    <div class="mini-card">
      <div class="mini-label">Advances</div>
      <div class="mini-value text-danger"><?php echo number_format($total_adv, 3); ?></div>
      <div class="mini-hint">Advance amounts already paid</div>
    </div>

    <div class="mini-card">
      <div class="mini-label">Shortages</div>
      <div class="mini-value text-danger"><?php echo number_format($total_short, 3); ?></div>
      <div class="mini-hint">COD and other recoveries</div>
    </div>
  </div>

  <div class="accordion" id="payoutAccordion">

    <!-- Earnings -->
    <div class="section-card">
      <div class="section-header" data-bs-toggle="collapse" data-bs-target="#earningsSection" aria-expanded="true">
        <div>
          <div class="section-title">Earnings</div>
          <div class="section-extra">Total <?php echo number_format($total_earnings, 3); ?> OMR</div>
        </div>
        <div class="chevron">&#9662;</div>
      </div>

      <div id="earningsSection" class="collapse show" data-bs-parent="#payoutAccordion">
        <div class="section-body">

          <div class="line-row" style="border-top:none;">
            <span class="line-label">Completed deliveries</span>
            <span class="line-amount pos"><?php echo number_format($completed_deliveries, 0); ?></span>
          </div>

          <table class="table table-sm mb-0">
            <tbody>
              <tr>
                <td class="text-muted">Operations fee</td>
                <td class="text-end text-success"><?php echo number_format($operations_fee, 3); ?></td>
              </tr>
              <tr>
                <td class="text-muted">Tips</td>
                <td class="text-end text-success"><?php echo number_format($tips, 3); ?></td>
              </tr>
              <tr>
                <td class="text-muted">Incentives</td>
                <td class="text-end text-success"><?php echo number_format($incentives, 3); ?></td>
              </tr>
              <tr>
                <td class="text-muted">Compensations</td>
                <td class="text-end text-success"><?php echo number_format($compensations, 3); ?></td>
              </tr>
              <tr>
                <td class="text-muted">Deductions</td>
                <td class="text-end text-danger"><?php echo number_format($deductions, 3); ?></td>
              </tr>
            </tbody>
          </table>

        </div>
      </div>
    </div>

    <!-- Adjustments -->
    <div class="section-card">
      <div class="section-header" data-bs-toggle="collapse" data-bs-target="#adjustSection">
        <div>
          <div class="section-title">Adjustments</div>
          <div class="section-extra">Net <?php echo number_format($net_adjustments, 3); ?> OMR</div>
        </div>
        <div class="chevron">&#9662;</div>
      </div>

      <div id="adjustSection" class="collapse" data-bs-parent="#payoutAccordion">
        <div class="section-body">
          <?php if (!$adjust_rows): ?>
            <div class="small text-muted py-2">No adjustments.</div>
          <?php else: ?>
            <?php foreach ($adjust_rows as $ar): ?>
              <?php
                $amount = (float)$ar['amount'];
                $signClass = $amount >= 0 ? 'pos' : 'neg';
                $categoryLabel = $ar['category_name'] ?: 'Adjustment';
                $reason = $ar['adj_remarks'] ?: $ar['label'];
                $label  = $categoryLabel . ' - ' . $reason;
              ?>
              <div class="line-row">
                <span class="line-label"><?php echo e($label); ?></span>
                <span class="line-amount <?php echo $signClass; ?>">
                  <?php echo number_format($amount, 3); ?>
                </span>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <!-- Advances -->
    <div class="section-card">
      <div class="section-header" data-bs-toggle="collapse" data-bs-target="#advanceSection">
        <div>
          <div class="section-title">Advances</div>
          <div class="section-extra">Total <?php echo number_format($total_adv, 3); ?> OMR</div>
        </div>
        <div class="chevron">&#9662;</div>
      </div>

      <div id="advanceSection" class="collapse" data-bs-parent="#payoutAccordion">
        <div class="section-body">
          <?php if (!$advance_rows): ?>
            <div class="small text-muted py-2">No advances.</div>
          <?php else: ?>
            <?php foreach ($advance_rows as $av): ?>
              <div class="line-row">
                <span class="line-label">Advance - <?php echo e($av['adj_remarks'] ?: $av['label']); ?></span>
                <span class="line-amount neg"><?php echo number_format((float)$av['amount'], 3); ?></span>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <!-- Shortages -->
    <div class="section-card">
      <div class="section-header" data-bs-toggle="collapse" data-bs-target="#shortageSection">
        <div>
          <div class="section-title">Shortages</div>
          <div class="section-extra">Total <?php echo number_format($total_short, 3); ?> OMR</div>
        </div>
        <div class="chevron">&#9662;</div>
      </div>

      <div id="shortageSection" class="collapse" data-bs-parent="#payoutAccordion">
        <div class="section-body">
          <?php if (!$shortage_rows): ?>
            <div class="small text-muted py-2">No shortages.</div>
          <?php else: ?>
            <?php foreach ($shortage_rows as $sr): ?>
              <div class="line-row">
                <span class="line-label">Shortage - <?php echo e($sr['adj_remarks'] ?: $sr['label']); ?></span>
                <span class="line-amount neg"><?php echo number_format((float)$sr['amount'], 3); ?></span>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>
      </div>
    </div>

  </div>
</div>

</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
