<?php
// public/driver/payout_slip.php

require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();

$cycle_id = (int)($_GET['cycle_id'] ?? 0);
if ($cycle_id <= 0) die('Missing payout cycle.');

$stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$cycle = $res->fetch_assoc();
$stmt->close();

if (!$cycle) die('Payout cycle not found.');

$sql = "
    SELECT pi.*, d.first_name, d.last_name
    FROM payout_items pi
    JOIN drivers d ON d.id = pi.driver_id
    WHERE pi.cycle_id = ? AND pi.driver_id = ?
    LIMIT 1
";
$stmt = $conn->prepare($sql);
$stmt->bind_param('ii', $cycle_id, $driver_id);
$stmt->execute();
$res  = $stmt->get_result();
$item = $res->fetch_assoc();
$stmt->close();

if (!$item) die('No payout item for this driver.');

$finance_note = trim($item['finance_note'] ?? '');

$earnSQL = "
    SELECT completed_deliveries, operations_fee, tips, incentives, compensations, deductions
    FROM driver_earnings_monthly
    WHERE driver_id = ? AND month = ?
    LIMIT 1
";
$stmtE = $conn->prepare($earnSQL);
$stmtE->bind_param('is', $driver_id, $cycle['cycle_month']);
$stmtE->execute();
$resE  = $stmtE->get_result();
$earn  = $resE->fetch_assoc();
$stmtE->close();

$completed_deliveries = (int)($earn['completed_deliveries'] ?? 0);
$operations_fee       = (float)($earn['operations_fee'] ?? 0);
$tips                 = (float)($earn['tips'] ?? 0);
$incentives           = (float)($earn['incentives'] ?? 0);
$compensations        = (float)($earn['compensations'] ?? 0);
$deductions           = (float)($earn['deductions'] ?? 0);

$total_earnings = (float)$item['total_earnings'];
$total_pos      = (float)$item['total_positive_adjustments'];
$total_neg      = (float)$item['total_negative_adjustments'];
$total_adv      = (float)$item['total_advances'];
$total_short    = (float)$item['total_shortages'];
$net_payout     = (float)$item['net_payout'];

$net_adjustments = $total_pos - $total_neg;

$stmt = $conn->prepare("
    SELECT pb.*, da.remarks AS adj_remarks,
           da.is_advance AS adj_is_advance,
           da.is_shortage AS adj_is_shortage,
           ac.name AS category_name
    FROM payout_breakdowns pb
    LEFT JOIN driver_adjustments da
      ON pb.reference_type = 'adjustment' AND pb.reference_id = da.id
    LEFT JOIN adjustment_categories ac
      ON da.category_id = ac.id
    WHERE pb.payout_item_id = ?
    ORDER BY pb.created_at ASC, pb.id ASC
");
$stmt->bind_param('i', $item['id']);
$stmt->execute();
$res = $stmt->get_result();

$adjust_rows   = [];
$advance_rows  = [];
$shortage_rows = [];

while ($r = $res->fetch_assoc()) {
    if ($r['type'] === 'earning') continue;

    $is_adv  = (int)($r['adj_is_advance'] ?? 0) === 1;
    $is_shrt = (int)($r['adj_is_shortage'] ?? 0) === 1;

    if ($is_adv)       $advance_rows[]  = $r;
    elseif ($is_shrt)  $shortage_rows[] = $r;
    else               $adjust_rows[]   = $r;
}

$stmt->close();

$driver_name = trim(($item['first_name'] ?? '') . ' ' . ($item['last_name'] ?? ''));
if ($driver_name === '') $driver_name = 'Driver';

?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Payout slip | Hisbuu</title>
<meta name="viewport" content="width=device-width, initial-scale=1">

<style>
    body { font-family: system-ui, sans-serif; background:#f3f4f6; margin:0; padding:0; }

    .slip-wrap {
        max-width: 900px;
        margin: 24px auto;
        background: #fff;
        border: 1px solid #d1d5db;
        border-radius: 8px;
        padding: 20px 26px;
    }

    .brand { font-size: 18px; font-weight: 700; color: #dc3545; }

    .flex-row {
        display: flex;
        gap: 24px;
        margin-top: 10px;
        align-items: flex-start;
    }

    .col {
        flex: 1;
        background: #fafafa;
        border: 1px solid #e5e7eb;
        border-radius: 6px;
        padding: 12px 14px;
    }

    .section-title { font-size: 14px; font-weight: 600; margin-bottom: 8px; }
    table { width: 100%; border-collapse: collapse; font-size: 13px; }
    td { padding: 4px 0; border-bottom: 1px solid #e5e7eb; }
    .text-right { text-align: right; }
    .pos { color: #16a34a; }
    .neg { color: #dc2626; }
    .total-row td { border-top: 2px solid #9ca3af; font-weight: 600; padding-top: 6px; }

    .footer { margin-top: 20px; font-size: 11px; text-align:right; color:#9ca3af; }

    @media print {
        body { background:#fff; }
        .slip-wrap { border:none; border-radius:0; width:100%; max-width:100%; padding:0; }
        .col { page-break-inside: avoid; }
    }
</style>
</head>

<body onload="window.print()">

<div class="slip-wrap">

    <div class="slip-header" style="display:flex; justify-content:space-between; margin-bottom:8px;">
        <div>
            <div class="brand">Hisbuu</div>
            <div class="slip-meta">Driver payout slip</div>
        </div>

        <div class="text-right">
            <div class="slip-title" style="font-weight:600;"><?php echo e($cycle['cycle_month']); ?></div>
            <div class="slip-meta">
                <?php echo e($cycle['period_start']); ?> → <?php echo e($cycle['period_end']); ?><br>
                Driver: <?php echo e($driver_name); ?>
            </div>
        </div>
    </div>

    <div class="section-title">Summary</div>
    <table>
        <tr><td>Earnings</td><td class="text-right pos"><?php echo number_format($total_earnings,3); ?> OMR</td></tr>

        <tr>
            <td>Net adjustments</td>
            <td class="text-right <?php echo $net_adjustments>=0?'pos':'neg'; ?>">
                <?php echo number_format($net_adjustments,3); ?> OMR
            </td>
        </tr>

        <tr><td>Advances</td><td class="text-right neg"><?php echo number_format($total_adv,3); ?> OMR</td></tr>
        <tr><td>Shortages</td><td class="text-right neg"><?php echo number_format($total_short,3); ?> OMR</td></tr>

        <tr class="total-row">
            <td>Net payout</td>
            <td class="text-right"><?php echo number_format($net_payout,3); ?> OMR</td>
        </tr>
    </table>

    <!-- TWO COLUMN BLOCK -->
    <div class="flex-row">

        <!-- Earnings breakdown -->
        <div class="col">
            <div class="section-title">Earnings breakdown</div>
            <table>
                <tr><td>Completed deliveries</td><td class="text-right pos"><?php echo number_format($completed_deliveries,0); ?></td></tr>
                <tr><td>Operations fee</td><td class="text-right pos"><?php echo number_format($operations_fee,3); ?></td></tr>
                <tr><td>Tips</td><td class="text-right pos"><?php echo number_format($tips,3); ?></td></tr>
                <tr><td>Incentives</td><td class="text-right pos"><?php echo number_format($incentives,3); ?></td></tr>
                <tr><td>Compensations</td><td class="text-right pos"><?php echo number_format($compensations,3); ?></td></tr>
                <tr><td>Deductions</td><td class="text-right neg"><?php echo number_format($deductions,3); ?></td></tr>
            </table>
        </div>

        <!-- Adjustments -->
        <div class="col">
            <div class="section-title">Adjustments</div>
            <table>
                <?php if (!$adjust_rows): ?>
                    <tr><td class="text-muted">No adjustments.</td></tr>
                <?php else: ?>
                    <?php foreach ($adjust_rows as $ar): ?>
                        <?php
                            $amount = (float)$ar['amount'];
                            $signClass = $amount>=0?'pos':'neg';
                            $cat = $ar['category_name'] ?: 'Adjustment';
                            $reason = $ar['adj_remarks'] ?: $ar['label'];
                            $label = $cat.' — '.$reason;
                        ?>
                        <tr>
                            <td><?php echo e($label); ?></td>
                            <td class="text-right <?php echo $signClass; ?>"><?php echo number_format($amount,3); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </table>
        </div>
    </div>

    <!-- ADVANCES & SHORTAGES BLOCK -->
    <div class="flex-row">

        <div class="col">
            <div class="section-title">Advances</div>
            <table>
                <?php if (!$advance_rows): ?>
                    <tr><td class="text-muted">No advances.</td></tr>
                <?php else: ?>
                    <?php foreach ($advance_rows as $av): ?>
                    <tr>
                        <td><?php echo e('Advance — '.($av['adj_remarks'] ?: $av['label'])); ?></td>
                        <td class="text-right neg"><?php echo number_format((float)$av['amount'],3); ?></td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </table>
        </div>

        <div class="col">
            <div class="section-title">Shortages</div>
            <table>
                <?php if (!$shortage_rows): ?>
                    <tr><td class="text-muted">No shortages.</td></tr>
                <?php else: ?>
                    <?php foreach ($shortage_rows as $sr): ?>
                    <tr>
                        <td><?php echo e('Shortage — '.($sr['adj_remarks'] ?: $sr['label'])); ?></td>
                        <td class="text-right neg"><?php echo number_format((float)$sr['amount'],3); ?></td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </table>
        </div>

    </div>

    <?php if ($finance_note !== ''): ?>
        <div class="note-box" style="margin-top:14px; padding:10px; border:1px solid #fed7aa; background:#fff7ed;">
            <strong>Finance note:</strong><br>
            <?php echo nl2br(e($finance_note)); ?>
        </div>
    <?php endif; ?>

    <div class="footer">
        Generated by Hisbuu payout system on <?php echo date('Y-m-d H:i'); ?>
    </div>

</div>
</body>
</html>
