<?php
// public/driver/my_gear.php

require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$title = 'My Gear & Assets';

$sql = "
    SELECT
        da.id,
        da.assigned_at,
        da.expiry_date,
        da.status,
        ai.id AS asset_item_id,
        ai.name AS item_name,
        ai.category,
        ai.has_expiry
    FROM driver_assets da
    JOIN asset_items ai ON da.asset_item_id = ai.id
    WHERE da.driver_id = ?
    ORDER BY da.expiry_date ASC, da.assigned_at DESC
    ";

$stmt = $conn->prepare($sql);
$stmt->bind_param('i', $driver_id);
$stmt->execute();
$assets = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Group assets by quantity for non-expiring items
$expiring_assets = [];
$grouped_assets = [];
foreach ($assets as $asset) {
    // Items with expiry dates are always listed individually
    if ($asset['has_expiry']) {
        $asset['quantity'] = 1;
        $asset['assignments'] = [['assigned_at' => $asset['assigned_at']]];
        $expiring_assets[] = $asset;
        continue;
    }
    
    // Group non-expiring items by their master ID
    $key = $asset['asset_item_id'];
    if (!isset($grouped_assets[$key])) {
        $grouped_assets[$key] = $asset;
        $grouped_assets[$key]['quantity'] = 1;
        $grouped_assets[$key]['assignments'] = [['assigned_at' => $asset['assigned_at']]];
    } else {
        $grouped_assets[$key]['quantity']++;
        $grouped_assets[$key]['assignments'][] = ['assigned_at' => $asset['assigned_at']];
    }
}

// Combine expiring and grouped assets into a single list for display
$final_assets = array_merge($expiring_assets, array_values($grouped_assets));

// Sort final list to ensure a logical order
usort($final_assets, function($a, $b) {
    // Items with expiry dates always come first
    if ($a['has_expiry'] && !$b['has_expiry']) return -1;
    // If both have expiry, sort by soonest date
    if ($a['has_expiry'] && $b['has_expiry']) {
        return strtotime($a['expiry_date']) <=> strtotime($b['expiry_date']);
    }
    // If neither have expiry (grouped items), sort by category then name
    return [$a['category'], $a['item_name']] <=> [$b['category'], $b['item_name']];
});

function get_expiry_badge(?string $expiry_date = null): string {
    if (!$expiry_date) {
        return '';
    }

    try {
        $expiry = new DateTimeImmutable($expiry_date);
    } catch (Exception $e) {
        return '';
    }

    $now = new DateTimeImmutable('now');

    // If the value is a plain date, treat it as expiring at the end of that day.
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', trim($expiry_date))) {
        $expiry = $expiry->setTime(23, 59, 59);
    }

    if ($expiry < $now) {
        return '<span class="badge text-bg-danger">Expired</span>';
    }

    $days = (int) $now->diff($expiry)->days;
    if ($days <= 30) {
        return '<span class="badge text-bg-warning">Expires Soon</span>';
    }

    return '';
}

require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<div class="d-flex justify-content-between align-items-center mb-3">
   <div>
     <h5 class="mb-0"><?php echo $title; ?></h5>
     <div class="text-muted small">A list of all company assets assigned to you.</div>
   </div>
   <a class="btn btn-outline-secondary btn-sm" href="<?php echo base_url('driver/dashboard.php'); ?>">Back to Dashboard</a>
</div>

<?php if (empty($final_assets)): ?>
   <div class="alert alert-light text-center">You have no assets assigned to you at the moment.</div>
<?php else: ?>
   <div class="accordion" id="assetsAccordion">
     <?php foreach ($final_assets as $asset):
       // Ensure a unique ID for the accordion element
       $accordionId = $asset['has_expiry'] ? 'asset-'.$asset['id'] : 'asset-group-'.$asset['asset_item_id'];
       ?>
        <div class="card mb-2">
          <div class="card-header p-0" id="heading-<?php echo $accordionId; ?>">
            <div class="d-flex align-items-center p-3">
              <div class="flex-grow-1">
                <div class="row align-items-center">
                 <div class="col-12 col-md-5">
                   <div class="small text-muted">Item Name</div>
                   <div class="fw-semibold"><?php echo e($asset['item_name']); ?></div>
                 </div>
                 <div class="col-6 col-md-2">
                   <div class="small text-muted">Category</div>
                   <div class="fw-semibold"><?php echo e($asset['category']); ?></div>
                 </div>
                 <?php if ($asset['quantity'] > 1): ?>
                   <div class="col-6 col-md-2">
                       <div class="small text-muted">Quantity</div>
                       <div class="fw-semibold"><?php echo $asset['quantity']; ?></div>
                   </div>
                 <?php endif; ?>
                 <div class="col-6 col-md-3">
                   <div class="small text-muted">Status</div>
                   <div>
                     <span class="badge text-bg-info"><?php echo e(ucfirst($asset['status'])); ?></span>
                     <?php echo get_expiry_badge($asset['expiry_date']); ?>
                   </div>
                 </div>
               </div>
             </div>
              <button class="btn btn-sm btn-outline-secondary ms-2" type="button" data-bs-toggle="collapse" data-bs-target="#<?php echo $accordionId; ?>" aria-expanded="false" aria-controls="<?php echo $accordionId; ?>">
                Details
              </button>
            </div>
          </div>
          <div id="<?php echo $accordionId; ?>" class="collapse" aria-labelledby="heading-<?php echo $accordionId; ?>" data-bs-parent="#assetsAccordion">
            <div class="card-body border-top">
              <div class="row">
                <div class="col-12 col-md-5">
                  <div class="small text-muted">Assignment Dates</div>
                  <ul class="list-unstyled mb-0">
                    <?php foreach ($asset['assignments'] as $assignment): ?>
                      <li><?php echo e(format_date($assignment['assigned_at'])); ?></li>
                    <?php endforeach; ?>
                  </ul>
                </div>
                <?php if ($asset['expiry_date']): ?>
                  <div class="col-12 col-md-4">
                    <div class="small text-muted">Expires On</div>
                    <p><?php echo e(format_date($asset['expiry_date'], 'Y-m-d')); ?></p>
                  </div>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
 <?php endif; ?>

</div>

<script>
document.querySelectorAll('form').forEach(f => {
    f.addEventListener('submit', e => {
        const b = e.submitter;
        if(b) { b.innerHTML='<span class="spinner-border spinner-border-sm"></span>'; b.disabled=true; }
    })
});
</script>

<?php
require_once __DIR__ . '/../../includes/driver_ui_end.php';
?>

