/* Hisbuu Driver - Service Worker */

const CACHE_VERSION = 'hisbuu-driver-v3';
const CACHE_NAME = CACHE_VERSION;

const APP_SHELL = [
  '/public/driver/',
  '/public/driver/dashboard.php',
  '/public/driver/offline.html',
  '/public/driver/manifest.json',
  '/public/driver/icons/icon-192.png',
  '/public/driver/icons/icon-512.png'
];

// Install - cache core shell
self.addEventListener('install', event => {
  event.waitUntil(
    caches.open(CACHE_NAME).then(cache => cache.addAll(APP_SHELL)).then(() => {
      return self.skipWaiting();
    })
  );
});

// Activate - clean old caches
self.addEventListener('activate', event => {
  event.waitUntil(
    caches.keys().then(keys =>
      Promise.all(
        keys
          .filter(key => key.startsWith('hisbuu-driver-') && key !== CACHE_NAME)
          .map(key => caches.delete(key))
      )
    ).then(() => self.clients.claim())
  );
});

// Handle fetch
self.addEventListener('fetch', event => {
  const req = event.request;

  // Only handle same-origin http(s)
  if (req.method !== 'GET' || !req.url.startsWith(self.location.origin)) {
    return;
  }

  // Navigation requests - show offline page if needed
  if (req.mode === 'navigate') {
    event.respondWith(
      fetch(req)
        .then(res => {
          return res;
        })
        .catch(() => {
          return caches.match('/public/driver/offline.html');
        })
    );
    return;
  }

  // Static assets - cache first, then network
  event.respondWith(
    caches.match(req).then(cached => {
      if (cached) {
        // Try to update in background
        fetch(req).then(res => {
          caches.open(CACHE_NAME).then(cache => cache.put(req, res.clone()));
        }).catch(() => {});
        return cached;
      }

      return fetch(req)
        .then(res => {
          const copy = res.clone();
          caches.open(CACHE_NAME).then(cache => cache.put(req, copy));
          return res;
        })
        .catch(() => {
          return new Response('', { status: 504, statusText: 'Offline' });
        });
    })
  );
});

// Allow page to ask SW to skip waiting
self.addEventListener('message', event => {
  if (!event.data) return;
  if (event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  }
});
