<?php
// public/driver/dashboard.php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$title = 'Dashboard';

$today = (new DateTime('today'))->format('Y-m-d');
$currentMonth = date('Y-m');
$filterMonth = trim($_GET['month'] ?? $currentMonth);
if (!preg_match('/^\d{4}-\d{2}$/', $filterMonth)) {
  $filterMonth = $currentMonth;
}
$monthStart = $filterMonth . '-01';
$monthEnd = date('Y-m-t', strtotime($monthStart));
$monthLabel = DateTime::createFromFormat('Y-m-d', $monthStart);
$monthLabel = $monthLabel ? $monthLabel->format('F Y') : $filterMonth;

// Driver basics
$driver = ['partner_id'=>'','first_name'=>'','last_name'=>''];
$st = $conn->prepare("SELECT partner_id, first_name, last_name FROM drivers WHERE id=?");
$st->bind_param('i', $driver_id);
$st->execute();
$driver = $st->get_result()->fetch_assoc() ?: $driver;

// Monthly metrics
$metrics = ['orders'=>0,'collection'=>0.0,'deposit'=>0.0,'balance'=>0.0];
$pd = $conn->prepare("
  SELECT COALESCE(SUM(CASE WHEN orders_count > 0 THEN orders_count ELSE total_orders END),0) AS total_orders,
         COALESCE(SUM(CASE WHEN amount_total <> 0 THEN amount_total ELSE total_amount END),0) AS total_amount
  FROM partner_daily
  WHERE driver_id=? AND service_date BETWEEN ? AND ?
");
$pd->bind_param('iss', $driver_id, $monthStart, $monthEnd);
$pd->execute();
if ($agg = $pd->get_result()->fetch_assoc()) {
  $metrics['orders'] = (int)($agg['total_orders'] ?? 0);
  $metrics['collection'] = (float)($agg['total_amount'] ?? 0);
}
$pd->close();

$dep = $conn->prepare("
  SELECT COALESCE(SUM(cash_deposited),0) AS total_deposit
  FROM submissions
  WHERE driver_id=? AND LOWER(status)='approved' AND service_date BETWEEN ? AND ?
");
$dep->bind_param('iss', $driver_id, $monthStart, $monthEnd);
$dep->execute();
if ($row = $dep->get_result()->fetch_assoc()) {
  $metrics['deposit'] = (float)($row['total_deposit'] ?? 0);
}
$dep->close();
$metrics['balance'] = $metrics['collection'] - $metrics['deposit'];

// Yet to submit: partner rows w/o matching submission, past dates only
$sqlYet = "
  SELECT pd.service_date, pd.total_orders, pd.total_amount, pd.orders_count, pd.amount_total
  FROM partner_daily pd
  LEFT JOIN submissions s
    ON s.driver_id=pd.driver_id AND s.service_date=pd.service_date
  WHERE pd.driver_id=? AND s.id IS NULL AND pd.service_date < ?
  ORDER BY pd.service_date DESC
  LIMIT 60
";
$y = $conn->prepare($sqlYet);
$y->bind_param('is', $driver_id, $today);
$y->execute();
$yet_rows = $y->get_result()->fetch_all(MYSQLI_ASSOC);

// Pending approval: this driver's submissions waiting review
$sqlPend = "
  SELECT id, service_date, orders_count, cash_deposited, trx_no, created_at
  FROM submissions
  WHERE driver_id=? AND status='pending'
  ORDER BY service_date DESC, created_at DESC
  LIMIT 60
";
$p = $conn->prepare($sqlPend);
$p->bind_param('i', $driver_id);
$p->execute();
$pend_rows = $p->get_result()->fetch_all(MYSQLI_ASSOC);

// Recent submissions, any status
$sqlRec = "
  SELECT service_date, orders_count, cash_deposited, trx_no, status, created_at, is_trx_validated
  FROM submissions
  WHERE driver_id=? AND service_date BETWEEN ? AND ?
  ORDER BY service_date DESC, created_at DESC
  LIMIT 5
";
$r = $conn->prepare($sqlRec);
$r->bind_param('iss', $driver_id, $monthStart, $monthEnd);
$r->execute();
$recent_rows = $r->get_result()->fetch_all(MYSQLI_ASSOC);

// helpers
function n3($v){ return number_format((float)$v, 3, '.', ''); }
function badge($s){
  $s = strtolower((string)$s);
  if ($s==='approved') return '<span class="badge text-bg-success">Approved</span>';
  if ($s==='rejected') return '<span class="badge text-bg-danger">Rejected</span>';
  return '<span class="badge text-bg-warning text-dark">Pending</span>';
}

// UI start
if (is_file(__DIR__.'/../../includes/driver_ui_start.php')) {
  include __DIR__.'/../../includes/driver_ui_start.php';
} else {
  echo '<!doctype html><html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">';
  echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">';
  echo '<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">'; // Keep icons for nav
  echo '<title>Driver Dashboard</title></head><body class="bg-light"><div class="container py-3">';
}
?><div class="d-flex justify-content-between align-items-center mb-3">
  <div>
    <h5 class="mb-0">Welcome, <?php echo e(trim(($driver['first_name']??'').' '.($driver['last_name']??'')) ?: 'Driver'); ?></h5>
    <div class="text-muted small">ID <?php echo e($driver['partner_id'] ?: ''); ?></div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-secondary btn-sm d-flex align-items-center gap-1" href="history.php">
      <i class="bi bi-clock-history"></i> History
    </a>
  </div>
</div>

<!-- Month filter -->
<div class="card mb-3">
  <div class="card-body">
    <form class="row g-2 align-items-end" method="get">
      <div class="col-sm-4 col-md-3">
        <label class="form-label small text-uppercase text-muted mb-1">Month</label>
        <input type="month" name="month" class="form-control" value="<?php echo e($filterMonth); ?>">
      </div>
      <div class="col-sm-4 col-md-3">
        <button class="btn btn-outline-secondary mt-3 mt-sm-0">Apply</button>
        <?php if ($filterMonth !== $currentMonth): ?>
          <a class="btn btn-link btn-sm" href="dashboard.php">Reset</a>
        <?php endif; ?>
      </div>
      <div class="col-12 col-md text-md-end text-muted small">
        Showing data for <strong><?php echo e($monthLabel); ?></strong>.
      </div>
    </form>
  </div>
</div>

<!-- KPI cards -->
<div class="row row-cols-1 row-cols-sm-2 row-cols-lg-4 g-3 mb-4">
  <div class="col">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted text-uppercase small">Total orders</div>
        <div class="fs-4 fw-semibold"><?php echo number_format($metrics['orders']); ?></div>
        <div class="small text-muted"><?php echo e($monthLabel); ?></div>
      </div>
    </div>
  </div>
  <div class="col">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted text-uppercase small">Total collection</div>
        <div class="fs-4 fw-semibold">OMR <?php echo n3($metrics['collection']); ?></div>
        <div class="small text-muted">From partner upload</div>
      </div>
    </div>
  </div>
  <div class="col">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted text-uppercase small">Total deposit</div>
        <div class="fs-4 fw-semibold">OMR <?php echo n3($metrics['deposit']); ?></div>
        <div class="small text-muted">Approved submissions</div>
      </div>
    </div>
  </div>
  <div class="col">
    <div class="card shadow-sm h-100 border-0 <?php echo $metrics['balance']>=0?'border-success':'border-danger'; ?>">
      <div class="card-body">
        <div class="text-muted text-uppercase small">Balance</div>
        <div class="fs-4 fw-semibold <?php echo $metrics['balance']>=0?'text-success':'text-danger'; ?>">
          OMR <?php echo n3($metrics['balance']); ?>
        </div>
        <div class="small text-muted">Collection minus deposits</div>
      </div>
    </div>
  </div>
</div>

<!-- Quick actions -->
<div class="card mb-3">
  <div class="card-body d-flex flex-wrap gap-2">
    <a class="btn btn-danger" href="submit.php">Make a submission</a>
  </div>
</div>

<!-- Yet to submit -->
<div class="mb-3">
  <div class="d-flex justify-content-between align-items-center bg-danger text-white px-3 py-2 rounded-top">
    <span>Yet to submit</span>
    <span class="badge text-bg-light"><?php echo count($yet_rows); ?></span>
  </div>
  <?php if (!$yet_rows): ?>
    <div class="p-3 text-muted border border-top-0 rounded-bottom">Nothing pending. You are up to date.</div>
  <?php else: ?>
    <div class="list-group list-group-flush rounded-bottom">
    <?php foreach ($yet_rows as $row): ?>
      <div class="list-group-item d-flex justify-content-between align-items-center">
        <div>
          <div class="fw-semibold"><?php echo e($row['service_date']); ?></div>
          <div class="small text-muted">Orders: <?php echo (int)($row['orders_count'] ?: $row['total_orders']); ?> · OMR <?php echo n3($row['amount_total'] != 0 ? $row['amount_total'] : $row['total_amount']); ?></div>
        </div>
        <a class="btn btn-sm btn-danger" href="submit.php?date=<?php echo urlencode($row['service_date']); ?>">Submit</a>
      </div>
    <?php endforeach; ?>
    </div>
  <?php endif; ?>
</div>

<!-- Pending approval -->
<div class="mb-3">
  <div class="d-flex justify-content-between align-items-center bg-warning px-3 py-2 rounded-top">
    <span>Pending approval</span>
    <span class="badge text-bg-dark"><?php echo count($pend_rows); ?></span>
  </div>
  <?php if (!$pend_rows): ?>
    <div class="p-3 text-muted border border-top-0 rounded-bottom">No submissions are waiting for approval.</div>
  <?php else: ?>
    <div class="list-group list-group-flush rounded-bottom">
    <?php foreach ($pend_rows as $row): ?>
      <div class="list-group-item d-flex justify-content-between align-items-center">
        <div>
          <div class="fw-semibold"><?php echo e($row['service_date']); ?></div>
          <div class="small text-muted">Orders: <?php echo (int)$row['orders_count']; ?> · OMR <?php echo n3($row['cash_deposited']); ?></div>
        </div>
        <a class="btn btn-sm btn-outline-secondary" href="resubmit.php?id=<?php echo (int)$row['id']; ?>">Review</a>
      </div>
    <?php endforeach; ?>
    </div>
  <?php endif; ?>
</div>

<!-- Recent submissions -->
<div class="mb-4">
  <div class="d-flex justify-content-between align-items-center bg-light text-dark px-3 py-2 rounded-top border">
    <span>Recent submissions</span>
    <span class="badge text-bg-secondary"><?php echo count($recent_rows); ?></span>
  </div>
  <?php if (!$recent_rows): ?>
    <div class="p-3 text-muted border border-top-0 rounded-bottom">You have no submissions yet.</div>
  <?php else: ?>
    <div class="list-group list-group-flush rounded-bottom">
    <?php foreach ($recent_rows as $row): ?>
      <div class="list-group-item d-flex justify-content-between align-items-center">
        <div>
          <div class="fw-semibold"><?php echo e($row['service_date']); ?></div>
          <div class="small text-muted">
            <div>Orders: <?php echo (int)$row['orders_count']; ?> · OMR <?php echo n3($row['cash_deposited']); ?></div>
            <?php if (!empty($row['trx_no'])): ?>
              <div class="d-flex align-items-center gap-1">
                <span>TRX: <?php echo e($row['trx_no']); ?></span>
                <?php if (!empty($row['is_trx_validated'])): ?>
                  <i class="bi bi-check-circle-fill text-success" title="Validated"></i>
                <?php endif; ?>
              </div>
            <?php endif; ?>
          </div>
        </div>
        <?php echo badge($row['status']); ?>
      </div>
    <?php endforeach; ?>
    </div>
  <?php endif; ?>
</div>

<?php
if (is_file(__DIR__.'/../../includes/driver_ui_end.php')) {
  include __DIR__.'/../../includes/driver_ui_end.php';
} else {
  echo '</div><script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script></body></html>';
}
