﻿<?php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driverId = get_current_driver_id();
$driver = ['partner_id' => '', 'partner_uuid' => '', 'full_name' => ''];
$driverStmt = $conn->prepare("SELECT partner_id, partner_uuid, CONCAT(first_name,' ',last_name) AS full_name FROM drivers WHERE id=? LIMIT 1");
$driverStmt->bind_param('i', $driverId);
$driverStmt->execute();
if ($info = $driverStmt->get_result()->fetch_assoc()) {
    $driver = $info;
}
$driverStmt->close();

$currentMonth = date('Y-m');
$month = trim($_GET['month'] ?? $currentMonth);
if (!preg_match('/^\d{4}-\d{2}$/', $month)) {
    $month = $currentMonth;
}
$startDate = $month . '-01';
$endDate = date('Y-m-t', strtotime($startDate));

$typeFilter = in_array($_GET['type'] ?? '', ['positive', 'negative'], true) ? $_GET['type'] : '';
$categoryFilter = isset($_GET['category']) && ctype_digit($_GET['category']) ? (int)$_GET['category'] : 0;

$categoryOptions = [];
$resultCats = $conn->query("SELECT id, name, type FROM adjustment_categories WHERE is_active=1 ORDER BY type, name");
while ($row = $resultCats->fetch_assoc()) {
    $categoryOptions[] = $row;
}

$baseWhere = 'a.driver_id=? AND a.adj_date BETWEEN ? AND ?';
$baseParams = [$driverId, $startDate, $endDate];
$baseTypes = 'iss';

// Summary totals ignore type/category filters so drivers always see full month overview.
$totals = ['positive' => 0.0, 'negative' => 0.0];
$sumStmt = $conn->prepare("SELECT type, SUM(amount) AS total_amt FROM driver_adjustments a WHERE $baseWhere GROUP BY type");
$sumStmt->bind_param($baseTypes, ...$baseParams);
$sumStmt->execute();
$sumResult = $sumStmt->get_result();
while ($row = $sumResult->fetch_assoc()) {
    $key = strtolower($row['type']);
    if (isset($totals[$key])) {
        $totals[$key] = (float)$row['total_amt'];
    }
}
$sumStmt->close();
$net = $totals['positive'] - $totals['negative'];

$listWhere = $baseWhere;
$listTypes = $baseTypes;
$listParams = $baseParams;
if ($typeFilter) {
    $listWhere .= ' AND a.type=?';
    $listTypes .= 's';
    $listParams[] = $typeFilter;
}
if ($categoryFilter > 0) {
    $listWhere .= ' AND a.category_id=?';
    $listTypes .= 'i';
    $listParams[] = $categoryFilter;
}

$listSql = "
    SELECT a.id, a.type, a.adj_date, a.amount, a.remarks, a.created_at,
           c.name AS category_name
    FROM driver_adjustments a
    JOIN adjustment_categories c ON c.id = a.category_id
    WHERE $listWhere
    ORDER BY a.adj_date DESC, a.id DESC
";
$listStmt = $conn->prepare($listSql);
$listStmt->bind_param($listTypes, ...$listParams);
$listStmt->execute();
$listRows = $listStmt->get_result()->fetch_all(MYSQLI_ASSOC);
$listStmt->close();

function adjustment_badge(string $type): string {
    if ($type === 'positive') {
        return '<span class="badge text-bg-success">Positive</span>';
    }
    return '<span class="badge text-bg-danger">Negative</span>';
}

$title = 'My Adjustments';
require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <div>
    <h5 class="mb-0">My Adjustments</h5>
    <div class="text-muted small">Driver ID: <?php echo e($driver['partner_id'] ?? ''); ?></div>
  </div>
  <a class="btn btn-outline-secondary btn-sm" href="<?php echo base_url('driver/dashboard.php'); ?>">Back to Dashboard</a>
</div>

<!-- Filters -->
<div class="card mb-3">
  <div class="card-body">
    <form class="row g-3 align-items-end" method="get">
      <div class="col-lg-3 col-md-6">
        <label for="month-filter" class="form-label">Month</label>
        <input type="month" id="month-filter" name="month" class="form-control" value="<?php echo e($month); ?>">
      </div>
      <div class="col-lg-3 col-md-6">
        <label for="type-filter" class="form-label">Type</label>
        <select id="type-filter" name="type" class="form-select">
          <option value="">All</option>
          <option value="positive" <?php echo $typeFilter==='positive'?'selected':''; ?>>Positive</option>
          <option value="negative" <?php echo $typeFilter==='negative'?'selected':''; ?>>Negative</option>
        </select>
      </div>
      <div class="col-lg-3 col-md-6">
        <label for="category-filter" class="form-label">Category</label>
        <select id="category-filter" name="category" class="form-select">
          <option value="0">All</option>
          <?php foreach ($categoryOptions as $cat): ?>
            <option value="<?php echo (int)$cat['id']; ?>" <?php echo $categoryFilter===(int)$cat['id']?'selected':''; ?>>
              <?php echo e($cat['name']); ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-lg-3 col-md-6">
        <button class="btn btn-outline-secondary">Apply</button>
        <a class="btn btn-link btn-sm" href="adjustments.php">Reset</a>
      </div>
    </form>
  </div>
</div>

<!-- Summary Cards -->
<div class="row row-cols-1 row-cols-md-3 g-3 mb-4">
  <div class="col-md-4">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Positives</div>
        <div class="fs-4 fw-semibold text-success">OMR <?php echo number_format($totals['positive'], 3); ?></div>
        <div class="small text-muted">Bonuses, incentives, etc.</div>
      </div>
    </div>
  </div>
  <div class="col-md-4">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Negatives</div>
        <div class="fs-4 fw-semibold text-danger">OMR <?php echo number_format($totals['negative'], 3); ?></div>
        <div class="small text-muted">Deductions, penalties, etc.</div>
      </div>
    </div>
  </div>
  <div class="col-md-4">
    <div class="card shadow-sm h-100 border-0">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Net impact</div>
        <div class="fs-4 fw-semibold <?php echo $net>=0?'text-success':'text-danger'; ?>">OMR <?php echo number_format($net, 3); ?></div>
        <div class="small text-muted">Positive minus negative</div>
      </div>
    </div>
  </div>
</div>

<?php if (!$listRows): ?>
  <div class="alert alert-light text-center">No adjustments found for the selected filters.</div>
<?php else: ?>
  <div class="accordion" id="adjustmentsAccordion">
    <?php foreach ($listRows as $row):
      $accordionId = 'adj-'.$row['id'];
      $amountClass = $row['type']==='positive'?'text-success':'text-danger';
    ?>
    <div class="card mb-2">
      <div class="card-header p-0" id="heading-<?php echo $accordionId; ?>">
        <div class="d-flex align-items-center p-3">
          <div class="flex-grow-1">
            <div class="row">
              <div class="col-6 col-md-3">
                <div class="small text-muted">Date</div>
                <div class="fw-semibold"><?php echo e($row['adj_date']); ?></div>
              </div>
              <div class="col-6 col-md-4">
                <div class="small text-muted">Category</div>
                <div class="fw-semibold"><?php echo e($row['category_name']); ?></div>
              </div>
              <div class="col-6 col-md-3 mt-2 mt-md-0">
                <div class="small text-muted">Amount</div>
                <div class="fw-semibold <?php echo $amountClass; ?>">OMR <?php echo number_format((float)$row['amount'], 3); ?></div>
              </div>
              <div class="col-6 col-md-2 mt-2 mt-md-0">
                <div class="small text-muted">Type</div>
                <div><?php echo adjustment_badge($row['type']); ?></div>
              </div>
            </div>
          </div>
          <button class="btn btn-sm btn-outline-secondary ms-2" type="button" data-bs-toggle="collapse" data-bs-target="#<?php echo $accordionId; ?>" aria-expanded="false" aria-controls="<?php echo $accordionId; ?>">
            Details
          </button>
        </div>
      </div>
      <div id="<?php echo $accordionId; ?>" class="collapse" aria-labelledby="heading-<?php echo $accordionId; ?>" data-bs-parent="#adjustmentsAccordion">
        <div class="card-body border-top">
            <div class="row">
              <div class="col-md-6">
                <div class="small text-muted">Remarks</div>
                <p><?php echo $row['remarks'] ? e($row['remarks']) : '<span class="text-muted">None</span>'; ?></p>
              </div>
              <div class="col-md-6">
                <div class="small text-muted">Recorded On</div>
                <p><?php echo e(format_date($row['created_at'])); ?></p>
              </div>
            </div>
        </div>
      </div>
    </div>
    <?php endforeach; ?>
  </div>
<?php endif; ?>

<div class="text-muted small mt-3">
  Need help? Contact your supervisor if an adjustment looks incorrect.
</div>

<?php require_once __DIR__ . '/../../includes/driver_ui_end.php'; ?>
