<?php
// public/admin/payouts/payout_driver_view.php
require_once __DIR__ . '/../../../includes/admin_bootstrap.php';

$cycle_id  = (int)($_GET['cycle_id'] ?? 0);
$driver_id = (int)($_GET['driver_id'] ?? 0);
$export    = $_GET['export'] ?? '';

// Load cycle
$stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$cycle = $res->fetch_assoc();
$stmt->close();

if (!$cycle) {
    die('Payout cycle not found.');
}

// Load payout item + driver
$sql = "
    SELECT
      pi.*,
      d.first_name,
      d.last_name
    FROM payout_items pi
    JOIN drivers d ON d.id = pi.driver_id
    WHERE pi.cycle_id = ? AND pi.driver_id = ?
    LIMIT 1
";
$stmt = $conn->prepare($sql);
$stmt->bind_param('ii', $cycle_id, $driver_id);
$stmt->execute();
$res  = $stmt->get_result();
$item = $res->fetch_assoc();
$stmt->close();

if (!$item) {
    die('No payout item for this driver in this cycle.');
}

// Load breakdown rows + adjustment + category
$stmt = $conn->prepare("
    SELECT
      pb.*,
      da.remarks          AS adj_remarks,
      da.is_advance       AS adj_is_advance,
      da.is_shortage      AS adj_is_shortage,
      da.type             AS adj_type,
      ac.name             AS category_name
    FROM payout_breakdowns pb
    LEFT JOIN driver_adjustments da
      ON pb.reference_type = 'adjustment' AND pb.reference_id = da.id
    LEFT JOIN adjustment_categories ac
      ON da.category_id = ac.id
    WHERE pb.payout_item_id = ?
    ORDER BY pb.created_at ASC, pb.id ASC
");
$stmt->bind_param('i', $item['id']);
$stmt->execute();
$res   = $stmt->get_result();
$rows  = [];
while ($r = $res->fetch_assoc()) {
    $rows[] = $r;
}
$stmt->close();

$driver_name = trim(($item['first_name'] ?? '') . ' ' . ($item['last_name'] ?? ''));
if ($driver_name === '') {
    $driver_name = 'Driver #' . $driver_id;
}

// CSV export
if ($export === 'csv') {
    $filename = 'payout_driver_' . $driver_id . '_' . $cycle['cycle_month'] . '.csv';

    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Type', 'CategoryAndReason', 'Amount', 'CreatedAt']);

    foreach ($rows as $r) {
        $amount      = (float)$r['amount'];
        $signSymbol  = $amount >= 0 ? '(+)' : '(-)';
        $typeDisplay = $r['type'] . ' ' . $signSymbol;

        $baseCategory = 'Adjustment';
        if ($r['type'] === 'earning') {
            $baseCategory = 'Earning';
        } elseif ($r['type'] === 'advance') {
            $baseCategory = 'Advance';
        } elseif ($r['type'] === 'shortage') {
            $baseCategory = 'Shortage';
        }

        if (!empty($r['category_name'])) {
            $categoryLabel = $r['category_name'];
        } else {
            $categoryLabel = $baseCategory;
        }

        $reason    = $r['adj_remarks'] ?: $r['label'];
        $fullLabel = $categoryLabel . ' — ' . $reason;

        fputcsv($out, [
            $typeDisplay,
            $fullLabel,
            number_format($amount, 3),
            $r['created_at'],
        ]);
    }

    fclose($out);
    exit;
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Driver Payout | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body {
      background-color: #f3f4f6;
    }
    .page-wrap {
      max-width: 1200px;
      margin: 0 auto;
    }
    .summary-grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
      gap: 16px;
      margin-bottom: 24px;
    }
    .summary-card {
      background: #ffffff;
      border-radius: 12px;
      padding: 14px 16px;
      min-height: 90px;
      box-shadow: 0 1px 3px rgba(15, 23, 42, 0.08);
      display: flex;
      flex-direction: column;
      justify-content: space-between;
    }
    .summary-label {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: .05em;
      color: #6b7280;
      margin-bottom: 4px;
    }
    .summary-value {
      font-size: 20px;
      font-weight: 600;
    }
    .summary-hint {
      font-size: 12px;
      color: #9ca3af;
    }
    .table-modern thead {
      border-bottom: 1px solid #e5e7eb;
    }
    .table-modern thead th {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: .05em;
      color: #6b7280;
      border-bottom: none !important;
      background: #f9fafb;
    }
    .table-modern tbody tr:nth-of-type(odd) {
      background-color: #f9fafb;
    }
    .table-modern tbody tr:hover {
      background-color: #eef2ff;
    }
    .type-pill {
      font-size: 12px;
      padding: 2px 8px;
      border-radius: 999px;
      border: 1px solid transparent;
      display: inline-block;
    }
    .type-pill.pos {
      color: #16a34a;
      border-color: #bbf7d0;
      background: #f0fdf4;
    }
    .type-pill.neg {
      color: #dc2626;
      border-color: #fecaca;
      background: #fef2f2;
    }
    .earning-main {
      font-size: 14px;
      font-weight: 500;
      margin-bottom: 4px;
    }
    .earning-breakdown {
      margin-top: 2px;
    }
    .earning-row {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 2px 0;
      border-top: 1px solid #e5e7eb;
    }
    .earning-row:first-child {
      margin-top: 2px;
    }
    @media print {
      .no-print {
        display: none !important;
      }
      body {
        background: #ffffff;
      }
      .summary-card {
        box-shadow: none;
      }
    }
  </style>
</head>
<body>

<?php include __DIR__ . '/../../../includes/admin_nav.php'; ?>

<div class="container-fluid px-4 py-3">
  <div class="page-wrap">

    <div class="d-flex justify-content-between align-items-start mb-3 no-print">
      <div>
        <h5 class="mb-1"><?php echo e($driver_name); ?></h5>
        <div class="text-muted small">
          Cycle <?php echo e($cycle['cycle_month']); ?>
          · <?php echo e($cycle['period_start']); ?> to <?php echo e($cycle['period_end']); ?>
        </div>
      </div>
      <div class="d-flex flex-wrap gap-2">
        <a href="payout_cycle_view.php?id=<?php echo $cycle_id; ?>"
           class="btn btn-outline-secondary btn-sm">
          Back to cycle
        </a>
        <button type="button"
                class="btn btn-outline-secondary btn-sm"
                onclick="window.print();">
          Print / PDF
        </button>
        <a href="payout_driver_view.php?cycle_id=<?php echo $cycle_id; ?>&driver_id=<?php echo $driver_id; ?>&export=csv"
           class="btn btn-danger btn-sm">
          Export CSV
        </a>
      </div>
    </div>

    <!-- Summary cards -->
    <div class="summary-grid mb-4">
      <div class="summary-card">
        <div>
          <div class="summary-label">Earnings</div>
          <div class="summary-value text-success">
            <?php echo number_format((float)$item['total_earnings'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">Orders based earnings</div>
      </div>

      <div class="summary-card">
        <div>
          <div class="summary-label">Positive adjustments</div>
          <div class="summary-value text-success">
            <?php echo number_format((float)$item['total_positive_adjustments'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">Bonuses, corrections, manual positives</div>
      </div>

      <div class="summary-card">
        <div>
          <div class="summary-label">Negative adjustments</div>
          <div class="summary-value text-danger">
            <?php echo number_format((float)$item['total_negative_adjustments'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">Penalties, deductions</div>
      </div>

      <div class="summary-card">
        <div>
          <div class="summary-label">Advances</div>
          <div class="summary-value text-danger">
            <?php echo number_format((float)$item['total_advances'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">Salary advances already paid</div>
      </div>

      <div class="summary-card">
        <div>
          <div class="summary-label">Shortages</div>
          <div class="summary-value text-danger">
            <?php echo number_format((float)$item['total_shortages'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">COD or other recoveries</div>
      </div>

      <div class="summary-card">
        <div>
          <div class="summary-label">Net payout</div>
          <div class="summary-value text-primary">
            <?php echo number_format((float)$item['net_payout'], 3); ?>
          </div>
        </div>
        <div class="summary-hint">What this driver should receive for this cycle</div>
      </div>
    </div>

    <!-- Breakdown -->
    <div class="card mb-4">
      <div class="card-header d-flex justify-content-between align-items-center bg-white">
        <span class="fw-semibold">Breakdown</span>
      </div>
      <div class="table-responsive">
        <table class="table table-modern table-sm mb-0">
          <thead>
          <tr>
            <th style="width:140px;">Type</th>
            <th>Category — Reason</th>
            <th class="text-end" style="width:140px;">Amount</th>
            <th style="width:180px;">Created at</th>
          </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr>
              <td colspan="4" class="text-center text-muted py-3">
                No breakdown rows recorded.
              </td>
            </tr>
          <?php else: ?>
            <?php foreach ($rows as $r): ?>
              <?php
              $amount      = (float)$r['amount'];
              $isPositive  = $amount >= 0;
              $signSymbol  = $isPositive ? '(+)' : '(-)';
              $signClass   = $isPositive ? 'text-success' : 'text-danger';
              $pillClass   = $isPositive ? 'type-pill pos' : 'type-pill neg';
              $isEarning   = ($r['type'] === 'earning');

              // Base category
              $baseCategory = 'Adjustment';
              if ($r['type'] === 'earning') {
                  $baseCategory = 'Earning';
              } elseif ($r['type'] === 'advance') {
                  $baseCategory = 'Advance';
              } elseif ($r['type'] === 'shortage') {
                  $baseCategory = 'Shortage';
              }

              if (!empty($r['category_name'])) {
                  $categoryLabel = $r['category_name'];
              } else {
                  $categoryLabel = $baseCategory;
              }

              $reasonRaw = $r['adj_remarks'] ?: $r['label'];

              $earningLines   = [];
              $hasEarningGrid = false;
              $fullLabelMain  = '';

              if ($isEarning) {
                  // Expect pattern: "Monthly earnings (...): Ops 200.000, Tips 5.000, ..."
                  $mainText = trim($reasonRaw);
                  $restPart = '';

                  if (strpos($reasonRaw, ':') !== false) {
                      $parts    = explode(':', $reasonRaw, 2);
                      $mainText = trim($parts[0]);
                      $restPart = trim($parts[1]);
                  }

                  if ($restPart !== '') {
                      $segments = preg_split('/\s*,\s*/', $restPart);
                      foreach ($segments as $seg) {
                          $seg = trim($seg);
                          if ($seg === '') continue;

                          if (preg_match('/^(.*)\s+(-?\d+(?:\.\d+)?)$/', $seg, $m)) {
                              $labelSeg = trim($m[1], " .");
                              $valSeg   = $m[2];
                          } else {
                              $labelSeg = $seg;
                              $valSeg   = '';
                          }

                          $earningLines[] = [
                              'label' => $labelSeg,
                              'value' => $valSeg,
                          ];
                      }
                  }

                  $hasEarningGrid = count($earningLines) > 0;
                  $fullLabelMain  = $categoryLabel . ' — ' . $mainText;
              } else {
                  $fullLabelMain = $categoryLabel . ' — ' . $reasonRaw;
              }

              $typeDisplay = $r['type'] . ' ' . $signSymbol;
              ?>
              <tr>
                <td>
                  <span class="<?php echo $pillClass; ?>">
                    <?php echo e($typeDisplay); ?>
                  </span>
                </td>

                <td>
                  <?php if ($isEarning && $hasEarningGrid): ?>
                    <div class="earning-main"><?php echo e($fullLabelMain); ?></div>
                    <div class="earning-breakdown">
                      <?php foreach ($earningLines as $line): ?>
                        <?php
                          $val      = $line['value'];
                          $valClass = 'text-muted';
                          if ($val !== '') {
                              $valFloat = (float)$val;
                              if ($valFloat > 0) {
                                  $valClass = 'text-success';
                              } elseif ($valFloat < 0) {
                                  $valClass = 'text-danger';
                              }
                          }
                        ?>
                        <div class="earning-row">
                          <span class="small text-muted"><?php echo e($line['label']); ?></span>
                          <span class="small <?php echo $valClass; ?>">
                            <?php echo e($val); ?>
                          </span>
                        </div>
                      <?php endforeach; ?>
                    </div>
                  <?php else: ?>
                    <?php echo e($fullLabelMain); ?>
                  <?php endif; ?>
                </td>

                <td class="text-end <?php echo $signClass; ?>">
                  <?php echo number_format($amount, 3); ?>
                </td>

                <td><small class="text-muted"><?php echo e($r['created_at']); ?></small></td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>

<!-- Net payout footer row -->
<tfoot>
<tr style="background:#fdf2f2;">
    <th colspan="2" class="text-end fw-semibold" style="border-top:2px solid #e5e7eb;">
        Net payout
    </th>
    <th class="text-end text-primary fw-bold" style="font-size:16px; border-top:2px solid #e5e7eb;">
        <?php echo number_format((float)$item['net_payout'], 3); ?>
    </th>
    <th style="border-top:2px solid #e5e7eb;"></th>
</tr>
</tfoot>

</table>
</div>
</div> <!-- end card -->


  </div>
</div>

</body>
</html>
