<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

$errors   = [];
$messages = [];
$missing_partner_ids = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_earnings'])) {
    $month   = trim($_POST['month'] ?? '');
    $partner = trim($_POST['partner'] ?? '');

    if (!preg_match('/^\d{4}-\d{2}$/', $month)) {
        $errors[] = 'Invalid month. Use format YYYY-MM, for example 2025-10.';
    }

    if (empty($_FILES['file']['tmp_name'])) {
        $errors[] = 'Please choose a CSV file.';
    }

    if (!$errors) {
        $tmpPath = $_FILES['file']['tmp_name'];
        $name    = $_FILES['file']['name'];
        $ext     = strtolower(pathinfo($name, PATHINFO_EXTENSION));

        if ($ext !== 'csv') {
            $errors[] = 'For now, only CSV files are supported. Save your Excel as CSV and upload that.';
        } else {
            $handle = fopen($tmpPath, 'r');
            if (!$handle) {
                $errors[] = 'Could not open uploaded CSV file.';
            } else {
                $header = fgetcsv($handle, 0, ',');
                if (!$header) {
                    $errors[] = 'CSV file appears to be empty.';
                } else {
                    // map header names -> index
                    $headerMap = [];
                    foreach ($header as $idx => $col) {
                        $col = strtolower(trim($col));
                        $headerMap[$col] = $idx;
                    }

                    // required columns
                    $requiredCols = [
                        'driver_id',              // this is partner_id
                        'completed_deliveries',
                        'operations_fee',
                        'tips',
                        'incentives',
                        'compensations',
                        'deductions'
                    ];

                    foreach ($requiredCols as $colName) {
                        if (!array_key_exists($colName, $headerMap)) {
                            $errors[] = "Missing required column: {$colName}";
                        }
                    }

                    if (!$errors) {
                        $uploaded_by      = (int)$_SESSION['user']['id'];
                        $inserted         = 0;
                        $updated          = 0;
                        $skipped_missing  = 0;
                        $skipped_other    = 0;

                        // prepared statement for insert or update
                        $sqlInsert = "
                            INSERT INTO driver_earnings_monthly (
                                driver_id, month,
                                completed_deliveries,
                                operations_fee, tips, incentives, compensations, deductions,
                                total_earning, partner, uploaded_by, created_at
                            )
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                            ON DUPLICATE KEY UPDATE
                                completed_deliveries = VALUES(completed_deliveries),
                                operations_fee       = VALUES(operations_fee),
                                tips                 = VALUES(tips),
                                incentives           = VALUES(incentives),
                                compensations        = VALUES(compensations),
                                deductions           = VALUES(deductions),
                                total_earning        = VALUES(total_earning),
                                partner              = VALUES(partner),
                                uploaded_by          = VALUES(uploaded_by)
                        ";

                        $stmtInsert = $conn->prepare($sqlInsert);

                        // lookup statement to resolve partner_id -> drivers.id
                        $stmtDriver = $conn->prepare("
                            SELECT id
                            FROM drivers
                            WHERE partner_id = ?
                            LIMIT 1
                        ");

                        while (($row = fgetcsv($handle, 0, ',')) !== false) {
                            // skip blank lines
                            if (count(array_filter($row, 'strlen')) === 0) {
                                continue;
                            }

                            $partner_id_csv = trim($row[$headerMap['driver_id']] ?? '');

                            if ($partner_id_csv === '') {
                                $skipped_other++;
                                continue;
                            }

                            // resolve partner_id to internal driver.id
                            $stmtDriver->bind_param('s', $partner_id_csv);
                            $stmtDriver->execute();
                            $resDriver = $stmtDriver->get_result();
                            $driverRow = $resDriver->fetch_assoc();

                            if (!$driverRow) {
                                $skipped_missing++;
                                $missing_partner_ids[] = $partner_id_csv;
                                continue;
                            }

                            $driver_id = (int)$driverRow['id'];

                            $completed_deliveries = (int)($row[$headerMap['completed_deliveries']] ?? 0);
                            $operations_fee       = (float)($row[$headerMap['operations_fee']] ?? 0);
                            $tips                 = (float)($row[$headerMap['tips']] ?? 0);
                            $incentives           = (float)($row[$headerMap['incentives']] ?? 0);
                            $compensations        = (float)($row[$headerMap['compensations']] ?? 0);
                            $deductions           = (float)($row[$headerMap['deductions']] ?? 0);

                            // allow explicit total_earning in CSV, otherwise compute it
                            $total_earning = $operations_fee + $tips + $incentives + $compensations + $deductions;
                            if (array_key_exists('total_earning', $headerMap)) {
                                $total_earning = (float)($row[$headerMap['total_earning']] ?? $total_earning);
                            }

                            $partnerVal = $partner !== '' ? $partner : null;

                            // types: i, s, i, d, d, d, d, d, d, s, i  => 11 params
                            $types = 'isiddddddsi';
                            $stmtInsert->bind_param(
                                $types,
                                $driver_id,
                                $month,
                                $completed_deliveries,
                                $operations_fee,
                                $tips,
                                $incentives,
                                $compensations,
                                $deductions,
                                $total_earning,
                                $partnerVal,
                                $uploaded_by
                            );

                            $stmtInsert->execute();

                            // affected_rows: 1 = insert, 2 = update on duplicate
                            if ($stmtInsert->affected_rows === 1) {
                                $inserted++;
                            } elseif ($stmtInsert->affected_rows === 2) {
                                $updated++;
                            } else {
                                $skipped_other++;
                            }
                        }

                        $stmtInsert->close();
                        $stmtDriver->close();
                        fclose($handle);

                        $messages[] = "Import complete. Inserted: {$inserted}, Updated: {$updated}, Skipped (no matching driver): {$skipped_missing}, Other skipped: {$skipped_other}.";

                        if ($missing_partner_ids) {
                            $missing_partner_ids = array_values(array_unique($missing_partner_ids));
                            $list = implode(', ', $missing_partner_ids);
                            $errors[] = 'These partner_id values had no matching driver record: ' . $list;
                        }
                    }
                }
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Upload Monthly Earnings | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-3">

  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Upload Monthly Earnings</h5>
    <div></div>
  </div>

  <?php if ($errors): ?>
    <div class="alert alert-danger py-2">
      <?php foreach ($errors as $e) echo e($e) . '<br>'; ?>
    </div>
  <?php endif; ?>

  <?php if ($messages): ?>
    <div class="alert alert-success py-2">
      <?php foreach ($messages as $m) echo e($m) . '<br>'; ?>
    </div>
  <?php endif; ?>

  <div class="card mb-3">
    <div class="card-body">
      <form method="post" enctype="multipart/form-data" class="row g-3">
        <div class="col-md-3">
          <label class="form-label">Month (YYYY-MM)</label>
          <input type="text" name="month" class="form-control" placeholder="<?php echo date('Y-m'); ?>" required>
        </div>
        <div class="col-md-3">
          <label class="form-label">Partner</label>
          <input type="text" name="partner" class="form-control" placeholder="Talabat, TMdone, etc.">
        </div>
        <div class="col-md-4">
          <label class="form-label">CSV file</label>
          <input type="file" name="file" class="form-control" accept=".csv" required>
          <div class="form-text">
            Export your Excel as CSV with columns:
            driver_id, completed_deliveries, operations_fee, tips, incentives, compensations, deductions, optional total_earning.
            driver_id must match drivers.partner_id in Hisbuu.
          </div>
        </div>
        <div class="col-md-2 d-flex align-items-end">
          <button class="btn btn-danger w-100" name="upload_earnings">Upload</button>
        </div>
      </form>
    </div>
  </div>

</div>

</body>
</html>
