<?php
// public/admin/drivers_bulk_upload.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';



$role = strtolower(current_user_role() ?? '');

if (!in_array($role, ['super administrator','administrator','finance administrator'], true)) {

  http_response_code(403); exit('Forbidden');

}



if (session_status() === PHP_SESSION_NONE) session_start();

if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));

$csrf = $_SESSION['csrf'];



header('X-Frame-Options: SAMEORIGIN');



function download_template() {

  $name = 'drivers_template.csv';

  header('Content-Type: text/csv; charset=utf-8');

  header('Content-Disposition: attachment; filename="'.$name.'"');

  $out = fopen('php://output', 'w');

  fputcsv($out, [

    'first_name','last_name','project_prefix','id_digits','password',

    'city_name','zone_name','nationality_name','type_name','salary_scheme_name',

    'work_phone','personal_phone','partner_uuid','ref_note'

  ]);

  fputcsv($out, ['Ahmed','Al Balushi','TB','11070108','Pass@123','Muscat','Ghubra','Omani','Full-time','Scheme A','+9689xxxxxxx','','','Top performer']);

  fputcsv($out, ['Rizwan','Zafar','KD','12987654','Pass@123','Muscat','','Pakistani','Part-time','Scheme B','','+9689yyyyyyy','','']);

  fclose($out);

  exit;

}

if (isset($_GET['action']) && $_GET['action']==='template') download_template();



function get_id_by_name(mysqli $conn, string $table, string $name): ?int {

  if ($name==='') return null;

  $sql = "SELECT id FROM {$table} WHERE name = ? LIMIT 1";

  $st = $conn->prepare($sql);

  $st->bind_param('s', $name);

  $st->execute();

  $res = $st->get_result()->fetch_assoc();

  return $res ? (int)$res['id'] : null;

}

function get_zone_by_city_and_name(mysqli $conn, ?int $city_id, string $zone_name): ?int {

  if (!$city_id || $zone_name==='') return null;

  $st = $conn->prepare("SELECT id FROM zones WHERE city_id=? AND name=? LIMIT 1");

  $st->bind_param('is', $city_id, $zone_name);

  $st->execute();

  $res = $st->get_result()->fetch_assoc();

  return $res ? (int)$res['id'] : null;

}

function project_prefix_exists(mysqli $conn, string $prefix): bool {

  $st = $conn->prepare("SELECT id FROM projects WHERE prefix=? LIMIT 1");

  $st->bind_param('s', $prefix);

  $st->execute();

  return (bool)$st->get_result()->fetch_assoc();

}

function make_partner_id(string $prefix, string $digits): string {

  return $prefix.$digits; // no zero padding per your rule

}

function hmac_sign(string $data, string $key): string {

  return hash_hmac('sha256', $data, $key);

}



// Phase flags

$is_preview = ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['phase']) && $_POST['phase']==='preview');

$is_import  = ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['phase']) && $_POST['phase']==='import');



$preview_rows = [];   // rows with fields + status

$ok_rows      = [];   // only valid rows used for import

$stats = ['total'=>0,'ok'=>0,'bad'=>0];

$flash_ok=''; $flash_err='';



// PHASE 2: Import confirmed rows

if ($is_import && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {

  $payload = trim($_POST['payload'] ?? '');

  $sig     = trim($_POST['sig'] ?? '');

  if ($payload === '' || $sig === '') {

    $flash_err = 'Missing import payload.';

  } else {

    $expected = hmac_sign($payload, $csrf);

    if (!hash_equals($expected, $sig)) {

      $flash_err = 'Invalid or expired import token.';

    } else {

      $ok_rows = json_decode($payload, true);

      if (!is_array($ok_rows)) {

        $flash_err = 'Import payload malformed.';

      } else {

        $inserted = 0; $skipped = 0;

        foreach ($ok_rows as $r) {

          // Required (from preview stage)

          $partner_id = $r['partner_id'];

          $first_name = $r['first_name'];

          $last_name  = $r['last_name'];

          $project_id = (int)$r['project_id'];

          $city_id    = (int)$r['city_id'];

          $zone_id    = $r['zone_id'] ? (int)$r['zone_id'] : null;

          $nationality_id   = $r['nationality_id'] ? (int)$r['nationality_id'] : null;

          $type_id          = $r['type_id'] ? (int)$r['type_id'] : null;

          $salary_scheme_id = $r['salary_scheme_id'] ? (int)$r['salary_scheme_id'] : null;

          $hash = $r['password_hash'];



          $work_phone     = $r['work_phone'];

          $personal_phone = $r['personal_phone'];

          $ref_note       = $r['ref_note'];

          $partner_uuid   = $r['partner_uuid'];



          $ins = $conn->prepare("

            INSERT INTO drivers(

              partner_id, first_name, last_name, project_id, city_id, zone_id,

              nationality_id, type_id, salary_scheme_id,

              login_password, status, work_phone, personal_phone, ref_note, partner_uuid

            ) VALUES (?,?,?,?,?,?,?,?,?,?,'active',?,?,?,?)

          ");

          $ins->bind_param(

            'sssiiiiissssss',

            $partner_id, $first_name, $last_name,

            $project_id, $city_id, $zone_id, $nationality_id, $type_id, $salary_scheme_id,

            $hash, $work_phone, $personal_phone, $ref_note, $partner_uuid

          );

          if ($ins->execute()) $inserted++; else $skipped++;

        }

        $flash_ok = "Imported $inserted rows. Skipped $skipped.";

      }

    }

  }

}



// PHASE 1: Preview after CSV upload

if ($is_preview && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {

  if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK) {

    $flash_err = 'CSV upload failed.';

  } else {

    $fh = fopen($_FILES['csv']['tmp_name'], 'r');

    if (!$fh) { $flash_err = 'Cannot open uploaded file.'; }

    else {

      $has_header = isset($_POST['has_header']);

      $rownum = 0;



      // Small caches to reduce DB roundtrips

      $cache_ids = [

        'projects' => [], 'cities'=>[], 'nationalities'=>[], 'types'=>[], 'salary_schemes'=>[], 'zones'=>[]

      ];



      while (($cols = fgetcsv($fh)) !== false) {

        $rownum++;

        if ($has_header && $rownum===1) continue;



        $stats['total']++;

        $cols = array_pad($cols, 14, '');

        list(

          $first_name, $last_name, $project_prefix, $id_digits, $password,

          $city_name, $zone_name, $nationality_name, $type_name, $salary_scheme_name,

          $work_phone, $personal_phone, $partner_uuid, $ref_note

        ) = array_map('trim', $cols);



        $errors = [];



        if ($first_name==='') $errors[]='first_name missing';

        if ($project_prefix==='') $errors[]='project_prefix missing';

        if ($id_digits==='' || !preg_match('/^[0-9]+$/', $id_digits)) $errors[]='id_digits invalid';

        if ($password==='') $errors[]='password missing';

        if ($city_name==='') $errors[]='city_name missing';

        if ($work_phone!=='' && !preg_match('/^\+?[0-9]{6,20}$/', $work_phone)) $errors[]='work_phone invalid';

        if ($personal_phone!=='' && !preg_match('/^\+?[0-9]{6,20}$/', $personal_phone)) $errors[]='personal_phone invalid';



        // Project by prefix

        $project_id = null;

        if ($project_prefix!=='') {

          if (!isset($cache_ids['projects'][$project_prefix])) {

            $st = $conn->prepare("SELECT id FROM projects WHERE prefix=? LIMIT 1");

            $st->bind_param('s', $project_prefix);

            $st->execute();

            $p = $st->get_result()->fetch_assoc();

            $cache_ids['projects'][$project_prefix] = $p ? (int)$p['id'] : 0;

          }

          $project_id = $cache_ids['projects'][$project_prefix];

          if (!$project_id) $errors[] = 'project_prefix not found in Projects';

        }



        // City

        $city_id = null;

        if ($city_name!=='') {

          if (!isset($cache_ids['cities'][$city_name])) {

            $cache_ids['cities'][$city_name] = get_id_by_name($conn, 'cities', $city_name) ?: 0;

          }

          $city_id = $cache_ids['cities'][$city_name];

          if (!$city_id) $errors[]='city not found';

        }



        // Zone

        $zone_id = null;

        if ($city_id && $zone_name!=='') {

          $key = $city_id.'|'.$zone_name;

          if (!isset($cache_ids['zones'][$key])) {

            $cache_ids['zones'][$key] = get_zone_by_city_and_name($conn, $city_id, $zone_name) ?: 0;

          }

          $zone_id = $cache_ids['zones'][$key] ?: null;

        }



        // Optional lookups

        $nationality_id = null;

        if ($nationality_name!=='') {

          if (!isset($cache_ids['nationalities'][$nationality_name])) {

            $cache_ids['nationalities'][$nationality_name] = get_id_by_name($conn, 'nationalities', $nationality_name) ?: 0;

          }

          $nationality_id = $cache_ids['nationalities'][$nationality_name] ?: null;

        }

        $type_id = null;

        if ($type_name!=='') {

          if (!isset($cache_ids['types'][$type_name])) {

            $cache_ids['types'][$type_name] = get_id_by_name($conn, 'types', $type_name) ?: 0;

          }

          $type_id = $cache_ids['types'][$type_name] ?: null;

        }

        $salary_scheme_id = null;

        if ($salary_scheme_name!=='') {

          if (!isset($cache_ids['salary_schemes'][$salary_scheme_name])) {

            $cache_ids['salary_schemes'][$salary_scheme_name] = get_id_by_name($conn, 'salary_schemes', $salary_scheme_name) ?: 0;

          }

          $salary_scheme_id = $cache_ids['salary_schemes'][$salary_scheme_name] ?: null;

        }



        // Build partner_id

        $partner_id = make_partner_id($project_prefix, $id_digits);



        // Uniqueness checks

        if ($partner_id!=='') {

          $st = $conn->prepare("SELECT id FROM drivers WHERE partner_id=? LIMIT 1");

          $st->bind_param('s', $partner_id);

          $st->execute();

          if ($st->get_result()->fetch_assoc()) $errors[]='partner_id already exists';

        }

        if ($partner_uuid!=='') {

          $st = $conn->prepare("SELECT id FROM drivers WHERE partner_uuid=? LIMIT 1");

          $st->bind_param('s', $partner_uuid);

          $st->execute();

          if ($st->get_result()->fetch_assoc()) $errors[]='partner_uuid already exists';

        }



        $status = $errors ? 'error' : 'ok';

        if ($status==='ok') {

          $stats['ok']++;

          $ok_rows[] = [

            'partner_id'=>$partner_id,

            'first_name'=>$first_name,

            'last_name'=>$last_name,

            'project_id'=>$project_id,

            'city_id'=>$city_id,

            'zone_id'=>$zone_id,

            'nationality_id'=>$nationality_id,

            'type_id'=>$type_id,

            'salary_scheme_id'=>$salary_scheme_id,

            'password_hash'=> password_hash($password, PASSWORD_BCRYPT),

            'work_phone'=>$work_phone,

            'personal_phone'=>$personal_phone,

            'ref_note'=>$ref_note,

            'partner_uuid'=>$partner_uuid,

          ];

        } else {

          $stats['bad']++;

        }



        $preview_rows[] = [

          'row' => $rownum,

          'partner_id' => $partner_id,

          'first_name' => $first_name,

          'last_name'  => $last_name,

          'project_prefix' => $project_prefix,

          'id_digits'  => $id_digits,

          'city'       => $city_name,

          'zone'       => $zone_name,

          'nationality'=> $nationality_name,

          'type'       => $type_name,

          'scheme'     => $salary_scheme_name,

          'work_phone' => $work_phone,

          'personal_phone'=>$personal_phone,

          'partner_uuid'=>$partner_uuid,

          'ref_note'   => $ref_note,

          'status'     => $status,

          'errors'     => $errors,

        ];

      }

      fclose($fh);

    }

  }

}



// Prepare signed payload for import

$payload_json = $ok_rows ? json_encode($ok_rows) : '';

$payload_sig  = $payload_json ? hmac_sign($payload_json, $csrf) : '';

?>

<!doctype html>

<html>

<head>

  <meta charset="utf-8">

  <title>Bulk Upload Drivers | Hisbuu</title>

  <meta name="viewport" content="width=device-width, initial-scale=1">

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>

    .status-ok { color:#0a7; font-weight:600; }

    .status-err{ color:#c00; font-weight:600; }

    .mono{ font-family: ui-monospace, SFMono-Regular, Menlo, Consolas, "Liberation Mono", monospace; }

    .smallmuted{ font-size:.9rem; color:#6c757d; }

  </style>

</head>

<body class="bg-light">

<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>



<div class="container py-4" style="max-width: 1100px;">

  <div class="d-flex justify-content-between align-items-center mb-3">

    <h5 class="mb-0">Bulk Upload Drivers</h5>

    <div class="d-flex gap-2">

      <a class="btn btn-outline-secondary btn-sm" href="<?php echo base_url('admin/driver_list.php'); ?>">Back</a>

      <a class="btn btn-danger btn-sm" href="<?php echo base_url('admin/drivers_bulk_upload.php'); ?>?action=template">Download CSV template</a>

    </div>

  </div>



  <?php if($flash_ok): ?><div class="alert alert-success py-2"><?php echo e($flash_ok); ?></div><?php endif; ?>

  <?php if($flash_err): ?><div class="alert alert-danger py-2"><?php echo e($flash_err); ?></div><?php endif; ?>



  <div class="card mb-3">

    <div class="card-body">

      <form method="post" enctype="multipart/form-data" class="row g-2">

        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">

        <input type="hidden" name="phase" value="preview">

        <div class="col-md-6">

          <label class="form-label">CSV file</label>

          <input type="file" name="csv" class="form-control" accept=".csv" required>

        </div>

        <div class="col-md-3 d-flex align-items-end">

          <div class="form-check">

            <input class="form-check-input" type="checkbox" name="has_header" id="has_header" checked>

            <label class="form-check-label" for="has_header">First row is header</label>

          </div>

        </div>

        <div class="col-md-3 d-flex align-items-end justify-content-end">

          <button class="btn btn-danger">Preview</button>

        </div>

      </form>

      <div class="smallmuted mt-2">

        Required: first_name, project_prefix, id_digits, password, city_name.  

        Optional: last_name, zone_name, nationality_name, type_name, salary_scheme_name, work_phone, personal_phone, partner_uuid, ref_note.

      </div>

    </div>

  </div>



  <?php if ($is_preview && $preview_rows): ?>

    <div class="alert alert-info py-2">

      Checked <?php echo (int)$stats['total']; ?> rows. OK <?php echo (int)$stats['ok']; ?>. Errors <?php echo (int)$stats['bad']; ?>.

      <button class="btn btn-success btn-sm ms-2" data-bs-toggle="modal" data-bs-target="#confirmImportModal" <?php echo $stats['ok'] ? '' : 'disabled'; ?>>

        Review & Import OK rows

      </button>

    </div>



    <div class="card mb-4">

      <div class="card-body">

        <div class="table-responsive">

          <table class="table table-sm align-middle">

            <thead class="table-light">

              <tr>

                <th>#</th>

                <th>Partner ID</th>

                <th>Name</th>

                <th>Project</th>

                <th>City</th>

                <th>Zone</th>

                <th>Phones</th>

                <th>Status</th>

                <th>Errors</th>

              </tr>

            </thead>

            <tbody>

              <?php foreach ($preview_rows as $r): ?>

                <tr>

                  <td><?php echo (int)$r['row']; ?></td>

                  <td class="mono"><?php echo e($r['partner_id']); ?></td>

                  <td><?php echo e(trim($r['first_name'].' '.$r['last_name'])); ?></td>

                  <td><?php echo e($r['project_prefix']); ?></td>

                  <td><?php echo e($r['city']); ?></td>

                  <td><?php echo e($r['zone'] ?: '-'); ?></td>

                  <td class="mono">

                    <?php echo e($r['work_phone'] ?: '-'); ?>

                    <?php if($r['personal_phone']): ?><br><?php echo e($r['personal_phone']); ?><?php endif; ?>

                  </td>

                  <td><?php echo $r['status']==='ok' ? '<span class="status-ok">OK</span>' : '<span class="status-err">ERROR</span>'; ?></td>

                  <td class="text-danger small">

                    <?php echo $r['errors'] ? e(implode('; ', $r['errors'])) : ''; ?>

                  </td>

                </tr>

              <?php endforeach; ?>

            </tbody>

          </table>

          <div class="smallmuted">

            Tip: fix CSV issues and re-upload to refresh this preview. Nothing is saved until you confirm import.

          </div>

        </div>

      </div>

    </div>



    <!-- Confirm Import Modal -->

    <div class="modal fade" id="confirmImportModal" tabindex="-1" aria-hidden="true">

      <div class="modal-dialog modal-xl modal-dialog-centered">

        <div class="modal-content">

          <form method="post">

            <div class="modal-header">

              <h6 class="modal-title">Import confirmation</h6>

              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>

            </div>

            <div class="modal-body">

              <p class="mb-2">These rows will be imported:</p>

              <div class="table-responsive" style="max-height:50vh">

                <table class="table table-sm">

                  <thead class="table-light">

                    <tr>

                      <th>Partner ID</th>

                      <th>Name</th>

                      <th>City</th>

                      <th>Zone</th>

                      <th>Nationality</th>

                      <th>Type</th>

                      <th>Scheme</th>

                      <th>Phones</th>

                    </tr>

                  </thead>

                  <tbody>

                    <?php foreach ($preview_rows as $r): if ($r['status']!=='ok') continue; ?>

                      <tr>

                        <td class="mono"><?php echo e($r['partner_id']); ?></td>

                        <td><?php echo e(trim($r['first_name'].' '.$r['last_name'])); ?></td>

                        <td><?php echo e($r['city']); ?></td>

                        <td><?php echo e($r['zone'] ?: '-'); ?></td>

                        <td><?php echo e($r['nationality'] ?: '-'); ?></td>

                        <td><?php echo e($r['type'] ?: '-'); ?></td>

                        <td><?php echo e($r['scheme'] ?: '-'); ?></td>

                        <td class="mono">

                          <?php echo e($r['work_phone'] ?: '-'); ?>

                          <?php if($r['personal_phone']): ?><br><?php echo e($r['personal_phone']); ?><?php endif; ?>

                        </td>

                      </tr>

                    <?php endforeach; ?>

                  </tbody>

                </table>

              </div>

              <div class="alert alert-warning small mt-3 mb-0">

                Once you click Import, only “OK” rows are inserted. Rows with errors are ignored.  

                To fix errors, close this modal, correct your CSV, and preview again.

              </div>

            </div>

            <div class="modal-footer">

              <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">

              <input type="hidden" name="phase" value="import">

              <input type="hidden" name="payload" value="<?php echo htmlspecialchars($payload_json, ENT_QUOTES, 'UTF-8'); ?>">

              <input type="hidden" name="sig" value="<?php echo htmlspecialchars($payload_sig, ENT_QUOTES, 'UTF-8'); ?>">

              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>

              <button class="btn btn-success" <?php echo $stats['ok'] ? '' : 'disabled'; ?>>Import</button>

            </div>

          </form>

        </div>

      </div>

    </div>

  <?php endif; ?>



</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

</body>

</html>

