<?php
require_once __DIR__.'/../../includes/admin_bootstrap.php';

header('Content-Type: application/json');

// Check permissions
if (!can_manage_drivers(get_current_role())) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Forbidden']);
    exit;
}

// Validate request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Validate CSRF token
$csrf = get_csrf_token();
if (!isset($_POST['csrf']) || !hash_equals($csrf, $_POST['csrf'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Invalid CSRF token']);
    exit;
}

// Get and validate parameters
$driver_id = isset($_POST['driver_id']) && ctype_digit($_POST['driver_id']) ? (int)$_POST['driver_id'] : 0;
$new_status = isset($_POST['new_status']) ? trim($_POST['new_status']) : '';

if ($driver_id <= 0) {
    echo json_encode(['success' => false, 'error' => 'Invalid driver ID']);
    exit;
}

if (!in_array($new_status, ['active', 'inactive'], true)) {
    echo json_encode(['success' => false, 'error' => 'Invalid status value']);
    exit;
}

// Check if driver exists
$stmt = $conn->prepare("SELECT id, partner_id, first_name, last_name, status FROM drivers WHERE id=? LIMIT 1");
$stmt->bind_param('i', $driver_id);
$stmt->execute();
$driver = $stmt->get_result()->fetch_assoc();

if (!$driver) {
    echo json_encode(['success' => false, 'error' => 'Driver not found']);
    exit;
}

// Update driver status
$update_stmt = $conn->prepare("UPDATE drivers SET status=? WHERE id=? LIMIT 1");
$update_stmt->bind_param('si', $new_status, $driver_id);

if ($update_stmt->execute()) {
    // Log the action
    $admin_id = current_user_id();
    $admin_name = current_user_name() ?? 'Unknown';
    $driver_name = trim(($driver['first_name'] ?? '') . ' ' . ($driver['last_name'] ?? ''));
    $old_status = $driver['status'];

    $log_msg = sprintf(
        'Driver %s (ID: %s) status changed from %s to %s by %s (ID: %d)',
        $driver_name,
        $driver['partner_id'],
        $old_status,
        $new_status,
        $admin_name,
        $admin_id
    );

    error_log($log_msg);

    // If deactivating, destroy any active driver sessions
    if ($new_status === 'inactive') {
        // This will be handled by the driver authentication check on their next request
        // We could also implement a more aggressive session invalidation if needed
    }

    echo json_encode([
        'success' => true,
        'new_status' => $new_status,
        'driver_id' => $driver_id,
        'message' => 'Driver status updated successfully'
    ]);
} else {
    echo json_encode(['success' => false, 'error' => 'Failed to update driver status']);
}
