<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

$role = strtolower(get_current_role() ?? '');
if (!in_array($role, ['super administrator', 'administrator', 'supervisor'], true)) {
  http_response_code(403);
  exit('Forbidden');
}

$user_id = (int)current_user_id();

if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$ok = '';
$err = '';

// Approve / Reject Document Set Logic
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  $set_id = isset($_POST['set_id']) && ctype_digit($_POST['set_id']) ? (int)$_POST['set_id'] : 0;
  $action = $_POST['action'] ?? '';

  if ($set_id > 0 && $action === 'approve_set') {
    $doc_number = trim($_POST['doc_number'] ?? '');
    $expiry = trim($_POST['expiry_date'] ?? '');

    $expiry_param = null;
    if ($expiry !== '') {
      $dt = DateTime::createFromFormat('Y-m-d', $expiry);
      if ($dt && $dt->format('Y-m-d') === $expiry) {
        $expiry_param = $expiry;
      } else {
        $err = 'Invalid expiry date format.';
      }
    }

    if (!$err) {
      $doc_num_param = $doc_number !== '' ? $doc_number : null;
      $stmtSet = $conn->prepare("UPDATE driver_doc_sets SET doc_number=?, expiry_date=?, status='approved', reviewer_id=?, reviewed_at=NOW() WHERE id=?");
      $stmtSet->bind_param('ssii', $doc_num_param, $expiry_param, $user_id, $set_id);

      if ($stmtSet->execute()) {
        // Also approve all documents in this set
        $stmtDocs = $conn->prepare("UPDATE driver_documents SET status='approved', reviewer_id=?, reviewed_at=NOW() WHERE doc_set_id=?");
        $stmtDocs->bind_param('ii', $user_id, $set_id);
        $stmtDocs->execute();
        $ok = 'Document set approved successfully.';
      } else {
        $err = 'Failed to approve document set.';
      }
    }
  }

  if ($set_id > 0 && $action === 'reject_set') {
    $reason = trim($_POST['reason'] ?? '');
    if ($reason === '') $reason = 'Rejected';

    $stmtSet = $conn->prepare("UPDATE driver_doc_sets SET status='rejected', reviewer_id=?, reviewed_at=NOW(), remarks=? WHERE id=?");
    $stmtSet->bind_param('isi', $user_id, $reason, $set_id);

    if ($stmtSet->execute()) {
      // Also reject all documents in this set
      $stmtDocs = $conn->prepare("UPDATE driver_documents SET status='rejected', reviewer_id=?, reviewed_at=NOW(), remarks=? WHERE doc_set_id=?");
      $stmtDocs->bind_param('isi', $user_id, $reason, $set_id);
      $stmtDocs->execute();
      $ok = 'Document set rejected.';
    } else {
      $err = 'Rejection failed.';
    }
  }
}

// Filters
$status = $_GET['status'] ?? 'pending';
$group_key = $_GET['group_key'] ?? '';
$q_driver = trim($_GET['driver'] ?? '');

$where = [];
$params = [];
$types = '';

if ($status !== 'all') { $where[] = "ds.status=?"; $params[] = $status; $types .= 's'; }
if ($group_key !== '') { $where[] = "ds.group_key=?"; $params[] = $group_key; $types .= 's'; }
if ($q_driver !== '') {
  $where[] = "(d.partner_id LIKE CONCAT('%',?,'%') OR CONCAT(d.first_name,' ',d.last_name) LIKE CONCAT('%',?,'%'))";
  $params[] = $q_driver; $params[] = $q_driver; $types .= 'ss';
}

$wsql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

// Pagination
$page = isset($_GET['page']) && ctype_digit($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 30;
$offset = ($page - 1) * $per_page;

// Get total count for pagination
$count_sql = "SELECT COUNT(*) as total FROM driver_doc_sets ds JOIN drivers d ON d.id = ds.driver_id $wsql";
$count_stmt = $conn->prepare($count_sql);
if ($types) $count_stmt->bind_param($types, ...$params);
$count_stmt->execute();
$total_count = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_count / $per_page);

// Query - Get Document Sets with their images
$sql = "
  SELECT ds.id AS set_id, ds.driver_id, ds.group_key, ds.doc_number, ds.expiry_date,
         ds.status, ds.remarks, ds.updated_at,
         d.partner_id, CONCAT(d.first_name, ' ', d.last_name) AS driver_name
  FROM driver_doc_sets ds
  JOIN drivers d ON d.id = ds.driver_id
  $wsql
  ORDER BY ds.updated_at DESC
  LIMIT ? OFFSET ?
";
$stmt = $conn->prepare($sql);
if ($types) {
  $stmt->bind_param($types . 'ii', ...array_merge($params, [$per_page, $offset]));
} else {
  $stmt->bind_param('ii', $per_page, $offset);
}
$stmt->execute();
$sets = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Fetch all documents for these sets
$set_docs = [];
if (!empty($sets)) {
  $set_ids = array_column($sets, 'set_id');
  $placeholders = implode(',', array_fill(0, count($set_ids), '?'));
  $types_docs = str_repeat('i', count($set_ids));

  $sql_docs = "SELECT doc_set_id, side, file_path, status FROM driver_documents WHERE doc_set_id IN ($placeholders)";
  $stmt_docs = $conn->prepare($sql_docs);
  $stmt_docs->bind_param($types_docs, ...$set_ids);
  $stmt_docs->execute();
  $docs_result = $stmt_docs->get_result();

  while ($doc = $docs_result->fetch_assoc()) {
    $set_docs[$doc['doc_set_id']][$doc['side']] = $doc;
  }
}

function badge_html($s) {
  $s = strtolower($s);
  if ($s === 'approved') return '<span class="badge text-bg-success">Approved</span>';
  if ($s === 'rejected') return '<span class="badge text-bg-danger">Rejected</span>';
  return '<span class="badge text-bg-warning text-dark">Pending</span>';
}

function doc_type_label($key) {
  $labels = [
    'civil_id' => 'Civil ID Card',
    'license' => 'Driving License',
    'vehicle_reg' => 'Vehicle Registration'
  ];
  return $labels[$key] ?? ucwords(str_replace('_', ' ', $key));
}

function doc_file_preview(?string $path, bool $thumbnail = true): string {
  if (!$path) return '<div class="text-muted small">Not uploaded</div>';
  $url = file_url($path);
  $ext = strtolower(pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION));
  if (in_array($ext, ['jpg','jpeg','png','gif','webp'], true)) {
    if ($thumbnail) {
      return '<img src="'.e($url).'" class="img-fluid rounded border doc-thumbnail" style="max-height: 200px; object-fit: contain; cursor: pointer;" alt="document" data-image-url="'.e($url).'">';
    }
    return '<img src="'.e($url).'" class="img-fluid rounded border" alt="document">';
  }
  if ($ext === 'pdf') {
    return '<div class="text-center p-3 border rounded"><i class="bi bi-file-pdf fs-1 text-danger"></i><br><small>PDF</small></div>';
  }
  return '<a href="'.e($url).'" target="_blank" class="btn btn-outline-secondary btn-sm">Open file</a>';
}
?>
<?php
$PAGE_TITLE = 'KYD Review (Sets)';
include __DIR__ . '/../../includes/admin_header.php';
?>

<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

<style>
.filter-card {
  background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
  color: white;
}
.filter-card .form-control,
.filter-card .form-select {
  border: 1px solid rgba(255,255,255,0.3);
  background: rgba(255,255,255,0.9);
}
.stats-bar {
  background: linear-gradient(to right, #f8f9fa, #ffffff);
  border-left: 4px solid #dc3545;
}
.doc-card {
  transition: all 0.2s;
  height: 100%;
}
.doc-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 8px 16px rgba(0,0,0,0.15);
}
.doc-thumbnail {
  cursor: pointer;
  transition: transform 0.2s;
}
.doc-thumbnail:hover {
  transform: scale(1.05);
}
.pagination .page-link {
  color: #dc3545;
  border-color: #dee2e6;
}
.pagination .page-link:hover {
  background-color: #fee;
  border-color: #dc3545;
}
.pagination .page-item.active .page-link {
  background-color: #dc3545;
  border-color: #dc3545;
}
</style>

<div class="d-flex justify-content-between align-items-start mb-4">
  <div>
    <h4 class="mb-1">KYD Document Review</h4>
    <p class="text-muted small mb-0">Review and approve driver document sets</p>
  </div>
</div>

<!-- Stats Bar -->
<div class="card stats-bar shadow-sm mb-4">
  <div class="card-body py-3">
    <div class="row align-items-center">
      <div class="col-auto">
        <i class="bi bi-file-earmark-check-fill fs-3 text-danger"></i>
      </div>
      <div class="col">
        <div class="row g-3">
          <div class="col-auto">
            <div class="small text-muted">Total Sets</div>
            <div class="h5 mb-0"><?= number_format($total_count) ?></div>
          </div>
          <div class="col-auto border-start">
            <div class="small text-muted">Current Page</div>
            <div class="h5 mb-0"><?= $page ?> / <?= max(1, $total_pages) ?></div>
          </div>
          <div class="col-auto border-start">
            <div class="small text-muted">Showing</div>
            <div class="h5 mb-0"><?= count($sets) ?> sets</div>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<?php if ($ok): ?>
<div class="alert alert-success alert-dismissible fade show shadow-sm">
  <i class="bi bi-check-circle-fill me-2"></i><?php echo e($ok); ?>
  <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<?php if ($err): ?>
<div class="alert alert-danger alert-dismissible fade show shadow-sm">
  <i class="bi bi-exclamation-triangle-fill me-2"></i><?php echo e($err); ?>
  <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<!-- Filters -->
<div class="card filter-card shadow-sm mb-4">
  <div class="card-body">
    <form method="get" class="row g-3">
      <div class="col-md-3">
        <label class="form-label small fw-bold">
          <i class="bi bi-funnel me-1"></i> Status
        </label>
        <select name="status" class="form-select">
          <?php foreach(['pending'=>'Pending','approved'=>'Approved','rejected'=>'Rejected','all'=>'All'] as $v=>$l): ?>
            <option value="<?php echo e($v); ?>" <?php echo $status===$v?'selected':''; ?>><?php echo e($l); ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-3">
        <label class="form-label small fw-bold">
          <i class="bi bi-file-earmark me-1"></i> Document Type
        </label>
        <select name="group_key" class="form-select">
          <option value="">All Types</option>
          <option value="civil_id" <?php echo $group_key==='civil_id'?'selected':''; ?>>Civil ID Card</option>
          <option value="license" <?php echo $group_key==='license'?'selected':''; ?>>Driving License</option>
          <option value="vehicle_reg" <?php echo $group_key==='vehicle_reg'?'selected':''; ?>>Vehicle Registration</option>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label small fw-bold">
          <i class="bi bi-search me-1"></i> Driver
        </label>
        <input type="text" name="driver" class="form-control" placeholder="Search by driver ID or name" value="<?php echo e($q_driver); ?>">
      </div>
      <div class="col-md-2 d-flex align-items-end">
        <button class="btn btn-light w-100">
          <i class="bi bi-funnel-fill me-1"></i> Apply Filters
        </button>
      </div>
    </form>
  </div>
</div>

  <?php if (!$sets): ?>
    <div class="text-center py-5">
      <i class="bi bi-inbox fs-1 text-muted"></i>
      <p class="text-muted mt-3 mb-0">No document sets found matching your criteria</p>
    </div>
  <?php else: ?>
    <div class="row row-cols-1 row-cols-lg-2 row-cols-xl-3 g-4">
        <?php foreach ($sets as $set):
          $set_id = (int)$set['set_id'];
          $docs = $set_docs[$set_id] ?? [];
          $front = $docs['front'] ?? null;
          $back = $docs['back'] ?? null;
        ?>
        <div class="col">
          <div class="card doc-card shadow-sm">
            <div class="card-header bg-white border-bottom">
              <div class="d-flex justify-content-between align-items-start">
                <div>
                  <h6 class="mb-1">
                    <i class="bi bi-file-earmark-text text-danger me-1"></i>
                    <?php echo e(doc_type_label($set['group_key'])); ?>
                  </h6>
                  <div class="text-muted small">
                    <i class="bi bi-person-badge me-1"></i>
                    <strong><?php echo e($set['partner_id']); ?></strong> - <?php echo e($set['driver_name']); ?>
                  </div>
                </div>
                <?php echo badge_html($set['status']); ?>
              </div>
            </div>
  
            <div class="card-body">
              <!-- Document Details -->
              <?php if ($set['doc_number'] || $set['expiry_date']): ?>
              <div class="mb-3 p-3 bg-light rounded border">
                <?php if ($set['doc_number']): ?>
                <div class="small mb-1">
                  <i class="bi bi-hash text-muted me-1"></i>
                  <strong>Number:</strong> <?php echo e($set['doc_number']); ?>
                </div>
                <?php endif; ?>
                <?php if ($set['expiry_date']): ?>
                <div class="small">
                  <i class="bi bi-calendar-event text-muted me-1"></i>
                  <strong>Expiry:</strong> <?php echo e(date('M d, Y', strtotime($set['expiry_date']))); ?>
                </div>
                <?php endif; ?>
              </div>
              <?php endif; ?>

              <!-- Front & Back Images Side by Side -->
              <div class="row g-2 mb-3">
                <div class="col-6">
                  <div class="text-center">
                    <div class="small fw-bold text-muted mb-2">
                      <i class="bi bi-arrow-right-circle me-1"></i>Front Side
                    </div>
                    <div class="border rounded p-2 bg-white position-relative">
                      <?php if ($front): ?>
                        <?php echo doc_file_preview($front['file_path'], true); ?>
                      <?php else: ?>
                        <div class="text-muted small p-3">
                          <i class="bi bi-image-alt"></i><br>Not uploaded
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                </div>
                <div class="col-6">
                  <div class="text-center">
                    <div class="small fw-bold text-muted mb-2">
                      <i class="bi bi-arrow-left-circle me-1"></i>Back Side
                    </div>
                    <div class="border rounded p-2 bg-white position-relative">
                      <?php if ($back): ?>
                        <?php echo doc_file_preview($back['file_path'], true); ?>
                      <?php else: ?>
                        <div class="text-muted small p-3">
                          <i class="bi bi-image-alt"></i><br>Not uploaded
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                </div>
              </div>

              <?php if ($set['remarks']): ?>
              <div class="alert alert-warning py-2 mb-2 small">
                <i class="bi bi-chat-left-text me-1"></i>
                <strong>Remarks:</strong> <?php echo e($set['remarks']); ?>
              </div>
              <?php endif; ?>

              <div class="text-muted small">
                <i class="bi bi-clock-history me-1"></i>
                Updated: <?php echo e(date('M d, Y g:i A', strtotime($set['updated_at']))); ?>
              </div>
            </div>
  
            <div class="card-footer bg-white border-top">
              <div class="d-flex gap-2">
                <?php if ($set['status'] !== 'approved'): ?>
                  <button class="btn btn-success btn-sm flex-fill" data-bs-toggle="modal" data-bs-target="#approve_<?php echo $set_id; ?>">
                    <i class="bi bi-check-circle me-1"></i> Approve
                  </button>
                <?php endif; ?>
                <?php if ($set['status'] !== 'rejected'): ?>
                  <button class="btn btn-danger btn-sm flex-fill" data-bs-toggle="modal" data-bs-target="#reject_<?php echo $set_id; ?>">
                    <i class="bi bi-x-circle me-1"></i> Reject
                  </button>
                <?php endif; ?>
                <button class="btn btn-outline-danger btn-sm" data-bs-toggle="modal" data-bs-target="#delete_<?php echo $set_id; ?>" title="Delete Document Set">
                  <i class="bi bi-trash"></i>
                </button>
              </div>
            </div>
          </div>
        </div>

        <?php endforeach; ?>
   </div>

   <!-- Pagination -->
   <?php if ($total_pages > 1): ?>
   <div class="d-flex justify-content-center mt-4">
     <nav>
       <ul class="pagination">
         <?php if ($page > 1): ?>
         <li class="page-item">
           <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
             <i class="bi bi-chevron-left"></i> Previous
           </a>
         </li>
         <?php endif; ?>

         <?php
         $start = max(1, $page - 2);
         $end = min($total_pages, $page + 2);

         if ($start > 1): ?>
           <li class="page-item"><a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => 1])) ?>">1</a></li>
           <?php if ($start > 2): ?><li class="page-item disabled"><span class="page-link">...</span></li><?php endif; ?>
         <?php endif; ?>

         <?php for ($i = $start; $i <= $end; $i++): ?>
         <li class="page-item <?= $i === $page ? 'active' : '' ?>">
           <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>"><?= $i ?></a>
         </li>
         <?php endfor; ?>

         <?php if ($end < $total_pages): ?>
           <?php if ($end < $total_pages - 1): ?><li class="page-item disabled"><span class="page-link">...</span></li><?php endif; ?>
           <li class="page-item"><a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $total_pages])) ?>"><?= $total_pages ?></a></li>
         <?php endif; ?>

         <?php if ($page < $total_pages): ?>
         <li class="page-item">
           <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
             Next <i class="bi bi-chevron-right"></i>
           </a>
         </li>
         <?php endif; ?>
       </ul>
     </nav>
   </div>
   <?php endif; ?>
 <?php endif; ?>

<!-- Image Preview Modal (Global) -->
<div class="modal fade" id="imagePreviewModal" tabindex="-1">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title"><i class="bi bi-image me-2"></i>Document Preview</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body text-center">
        <img id="previewImage" src="" class="img-fluid" alt="Document preview">
      </div>
      <div class="modal-footer">
        <a id="previewDownload" href="" download class="btn btn-primary">
          <i class="bi bi-download me-1"></i>Download
        </a>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Modals for each set -->
<?php foreach ($sets as $set):
  $set_id = (int)$set['set_id'];
  $docs = $set_docs[$set_id] ?? [];
  $front = $docs['front'] ?? null;
  $back = $docs['back'] ?? null;
?>
  <!-- Approve Modal -->
  <?php if ($set['status'] !== 'approved'): ?>
  <div class="modal fade" id="approve_<?php echo $set_id; ?>" tabindex="-1">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="post">
          <div class="modal-header bg-light">
            <h5 class="modal-title">
              <i class="bi bi-check-circle text-success me-2"></i>
              Approve: <?php echo e(doc_type_label($set['group_key'])); ?>
            </h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
          </div>
          <div class="modal-body">
            <div class="alert alert-info mb-4">
              <i class="bi bi-info-circle me-2"></i>
              <strong>Driver:</strong> <?php echo e($set['partner_id']); ?> - <?php echo e($set['driver_name']); ?>
            </div>

            <div class="row mb-4">
              <div class="col-md-6">
                <label class="form-label">
                  <i class="bi bi-hash me-1"></i>Document Number
                </label>
                <input class="form-control" name="doc_number" value="<?php echo e($set['doc_number'] ?? ''); ?>" placeholder="Enter document number">
              </div>
              <div class="col-md-6">
                <label class="form-label">
                  <i class="bi bi-calendar-event me-1"></i>Expiry Date
                </label>
                <input type="date" class="form-control" name="expiry_date" value="<?php echo e($set['expiry_date'] ?? ''); ?>">
              </div>
            </div>

            <div class="row g-4">
              <div class="col-md-6">
                <h6 class="text-muted mb-3">
                  <i class="bi bi-arrow-right-circle me-1"></i>Front Side
                </h6>
                <?php if ($front): ?>
                  <img src="<?php echo e(file_url($front['file_path'])); ?>" class="img-fluid border rounded shadow-sm" alt="front">
                <?php else: ?>
                  <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle me-2"></i>Not uploaded
                  </div>
                <?php endif; ?>
              </div>
              <div class="col-md-6">
                <h6 class="text-muted mb-3">
                  <i class="bi bi-arrow-left-circle me-1"></i>Back Side
                </h6>
                <?php if ($back): ?>
                  <img src="<?php echo e(file_url($back['file_path'])); ?>" class="img-fluid border rounded shadow-sm" alt="back">
                <?php else: ?>
                  <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle me-2"></i>Not uploaded
                  </div>
                <?php endif; ?>
              </div>
            </div>
          </div>
          <div class="modal-footer bg-light">
            <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
            <input type="hidden" name="set_id" value="<?php echo $set_id; ?>">
            <input type="hidden" name="action" value="approve_set">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
              <i class="bi bi-x-circle me-1"></i>Cancel
            </button>
            <button class="btn btn-success">
              <i class="bi bi-check-circle me-1"></i>Approve Document Set
            </button>
          </div>
        </form>
      </div>
    </div>
  </div>
  <?php endif; ?>
  
  <!-- Reject Modal -->
  <?php if ($set['status'] !== 'rejected'): ?>
  <div class="modal fade" id="reject_<?php echo $set_id; ?>" tabindex="-1">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="post">
          <div class="modal-header bg-danger bg-opacity-10">
            <h5 class="modal-title text-danger">
              <i class="bi bi-x-circle me-2"></i>
              Reject: <?php echo e(doc_type_label($set['group_key'])); ?>
            </h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
          </div>
          <div class="modal-body">
            <div class="alert alert-warning mb-4">
              <i class="bi bi-exclamation-triangle me-2"></i>
              <strong>Driver:</strong> <?php echo e($set['partner_id']); ?> - <?php echo e($set['driver_name']); ?>
              <br><small>The driver will be notified of the rejection and can resubmit.</small>
            </div>

            <div class="row g-4 mb-4">
              <div class="col-md-6">
                <h6 class="text-muted mb-3">
                  <i class="bi bi-arrow-right-circle me-1"></i>Front Side
                </h6>
                <?php if ($front): ?>
                  <img src="<?php echo e(file_url($front['file_path'])); ?>" class="img-fluid border rounded shadow-sm" alt="front">
                <?php else: ?>
                  <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle me-2"></i>Not uploaded
                  </div>
                <?php endif; ?>
              </div>
              <div class="col-md-6">
                <h6 class="text-muted mb-3">
                  <i class="bi bi-arrow-left-circle me-1"></i>Back Side
                </h6>
                <?php if ($back): ?>
                  <img src="<?php echo e(file_url($back['file_path'])); ?>" class="img-fluid border rounded shadow-sm" alt="back">
                <?php else: ?>
                  <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle me-2"></i>Not uploaded
                  </div>
                <?php endif; ?>
              </div>
            </div>

            <div class="border-top pt-4">
              <label class="form-label fw-bold text-danger">
                <i class="bi bi-chat-left-text me-1"></i>Rejection Reason *
              </label>
              <textarea name="reason" class="form-control" rows="4" required placeholder="Please provide a clear reason for rejecting this document set. Be specific about what needs to be corrected."></textarea>
              <div class="form-text">This message will be visible to the driver.</div>
            </div>
          </div>
          <div class="modal-footer bg-light">
            <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
            <input type="hidden" name="set_id" value="<?php echo $set_id; ?>">
            <input type="hidden" name="action" value="reject_set">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
              <i class="bi bi-x-circle me-1"></i>Cancel
            </button>
            <button class="btn btn-danger">
              <i class="bi bi-x-circle me-1"></i>Reject Document Set
            </button>
          </div>
        </form>
      </div>
    </div>
  </div>
  <?php endif; ?>

  <!-- Delete Modal -->
  <div class="modal fade" id="delete_<?php echo $set_id; ?>" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
      <div class="modal-content">
        <div class="modal-header bg-danger text-white">
          <h5 class="modal-title">
            <i class="bi bi-trash me-2"></i>
            Delete Document Set
          </h5>
          <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <div class="alert alert-danger mb-3">
            <i class="bi bi-exclamation-triangle-fill me-2"></i>
            <strong>Warning:</strong> This action cannot be undone!
          </div>

          <div class="mb-3">
            <h6>Document Set Details:</h6>
            <ul class="list-unstyled mb-0">
              <li><strong>Type:</strong> <?php echo e(doc_type_label($set['group_key'])); ?></li>
              <li><strong>Driver:</strong> <?php echo e($set['partner_id']); ?> - <?php echo e($set['driver_name']); ?></li>
              <li><strong>Status:</strong> <?php echo badge_html($set['status']); ?></li>
              <?php if ($set['doc_number']): ?>
              <li><strong>Doc Number:</strong> <?php echo e($set['doc_number']); ?></li>
              <?php endif; ?>
            </ul>
          </div>

          <p class="text-danger fw-bold mb-0">
            Are you sure you want to permanently delete this document set and all associated files?
          </p>
        </div>
        <div class="modal-footer bg-light">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
            <i class="bi bi-x-circle me-1"></i>Cancel
          </button>
          <button type="button" class="btn btn-danger delete-set-btn"
                  data-set-id="<?php echo $set_id; ?>"
                  data-driver-name="<?php echo e($set['driver_name']); ?>"
                  data-doc-type="<?php echo e(doc_type_label($set['group_key'])); ?>">
            <i class="bi bi-trash me-1"></i>Delete Permanently
          </button>
        </div>
      </div>
    </div>
  </div>

<?php endforeach; ?>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script>
$(function() {
  // Image preview modal
  $(document).on('click', '.doc-thumbnail', function() {
    const imageUrl = $(this).data('image-url');
    $('#previewImage').attr('src', imageUrl);
    $('#previewDownload').attr('href', imageUrl);
    const modal = new bootstrap.Modal(document.getElementById('imagePreviewModal'));
    modal.show();
  });

  // Delete document set
  $(document).on('click', '.delete-set-btn', function() {
    const $btn = $(this);
    const setId = $btn.data('set-id');
    const driverName = $btn.data('driver-name');
    const docType = $btn.data('doc-type');

    // Disable button during request
    $btn.prop('disabled', true).html('<i class="bi bi-hourglass-split"></i> Deleting...');

    $.ajax({
      url: '<?= base_url('admin/driver_doc_set_delete.php') ?>',
      method: 'POST',
      data: {
        set_id: setId,
        csrf: '<?= $csrf ?>'
      },
      dataType: 'json',
      success: function(response) {
        if (response.success) {
          // Close modal
          const modalId = '#delete_' + setId;
          const modal = bootstrap.Modal.getInstance(document.querySelector(modalId));
          if (modal) modal.hide();

          // Remove the card from the page with animation
          const $card = $btn.closest('.col');
          $card.fadeOut(400, function() {
            $(this).remove();

            // Show success message
            const alertHtml = '<div class="alert alert-success alert-dismissible fade show shadow-sm">' +
              '<i class="bi bi-check-circle-fill me-2"></i>' +
              'Document set for ' + driverName + ' (' + docType + ') has been deleted successfully.' +
              '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>' +
              '</div>';

            $('.stats-bar').after(alertHtml);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
              $('.alert-success').fadeOut('slow');
            }, 5000);

            // If no more cards, show empty state
            if ($('.doc-card').length === 0) {
              $('.row.row-cols-1').html(
                '<div class="col-12 text-center py-5">' +
                '<i class="bi bi-inbox fs-1 text-muted"></i>' +
                '<p class="text-muted mt-3 mb-0">No document sets found matching your criteria</p>' +
                '</div>'
              );
            }
          });
        } else {
          alert('Error: ' + (response.error || 'Failed to delete document set'));
          $btn.prop('disabled', false).html('<i class="bi bi-trash me-1"></i>Delete Permanently');
        }
      },
      error: function(xhr) {
        let errorMsg = 'An error occurred while deleting the document set.';
        try {
          const response = JSON.parse(xhr.responseText);
          if (response.error) errorMsg = response.error;
        } catch (e) {
          // Use default error message
        }
        alert(errorMsg);
        $btn.prop('disabled', false).html('<i class="bi bi-trash me-1"></i>Delete Permanently');
      }
    });
  });

  // Auto-dismiss alerts after 5 seconds
  setTimeout(function() {
    $('.alert-success, .alert-danger').fadeOut('slow');
  }, 5000);
});
</script>

<?php include __DIR__ . '/../../includes/admin_footer.php'; ?>
