<?php
require_once __DIR__.'/../../includes/admin_bootstrap.php';

header('Content-Type: application/json');

$role = strtolower(get_current_role() ?? '');
if (!in_array($role, ['super administrator', 'administrator', 'supervisor'], true)) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Forbidden']);
    exit;
}

// Validate request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Validate CSRF token
if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

if (!isset($_POST['csrf']) || !hash_equals($csrf, $_POST['csrf'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Invalid CSRF token']);
    exit;
}

// Get and validate parameters
$set_id = isset($_POST['set_id']) && ctype_digit($_POST['set_id']) ? (int)$_POST['set_id'] : 0;

if ($set_id <= 0) {
    echo json_encode(['success' => false, 'error' => 'Invalid document set ID']);
    exit;
}

// Fetch the document set details
$stmt = $conn->prepare("
    SELECT ds.id, ds.driver_id, ds.group_key, ds.doc_number, ds.status,
           d.partner_id, CONCAT(d.first_name, ' ', d.last_name) AS driver_name
    FROM driver_doc_sets ds
    JOIN drivers d ON d.id = ds.driver_id
    WHERE ds.id = ?
    LIMIT 1
");
$stmt->bind_param('i', $set_id);
$stmt->execute();
$set = $stmt->get_result()->fetch_assoc();

if (!$set) {
    echo json_encode(['success' => false, 'error' => 'Document set not found']);
    exit;
}

// Fetch all documents in this set to delete files
$docs_stmt = $conn->prepare("SELECT id, file_path FROM driver_documents WHERE doc_set_id = ?");
$docs_stmt->bind_param('i', $set_id);
$docs_stmt->execute();
$docs = $docs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Begin transaction
$conn->begin_transaction();

try {
    // Delete physical files from storage
    foreach ($docs as $doc) {
        if (!empty($doc['file_path'])) {
            // Extract path from storage
            $file_path = $doc['file_path'];

            // If path starts with 'storage/', it's already a relative path from STORAGE_DIR
            if (strpos($file_path, 'storage/') === 0) {
                $file_path = substr($file_path, 8); // Remove 'storage/' prefix
            }

            $full_path = STORAGE_DIR . '/' . $file_path;

            if (file_exists($full_path)) {
                @unlink($full_path);
                error_log("Deleted document file: {$full_path}");
            }
        }
    }

    // Delete document records
    $delete_docs = $conn->prepare("DELETE FROM driver_documents WHERE doc_set_id = ?");
    $delete_docs->bind_param('i', $set_id);
    $delete_docs->execute();

    // Delete document set record
    $delete_set = $conn->prepare("DELETE FROM driver_doc_sets WHERE id = ?");
    $delete_set->bind_param('i', $set_id);
    $delete_set->execute();

    // Commit transaction
    $conn->commit();

    // Log the action
    $admin_id = current_user_id();
    $admin_name = current_user_name() ?? 'Unknown';
    $log_msg = sprintf(
        'Document set #%d (%s) for driver %s (ID: %s) deleted by %s (ID: %d)',
        $set_id,
        $set['group_key'],
        $set['driver_name'],
        $set['partner_id'],
        $admin_name,
        $admin_id
    );
    error_log($log_msg);

    echo json_encode([
        'success' => true,
        'message' => 'Document set deleted successfully',
        'set_id' => $set_id
    ]);

} catch (Exception $e) {
    // Rollback on error
    $conn->rollback();

    error_log('Document set deletion failed: ' . $e->getMessage());

    echo json_encode([
        'success' => false,
        'error' => 'Failed to delete document set: ' . $e->getMessage()
    ]);
}
