<?php
// public/admin/data_entry.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';
$role = get_current_role();
ensure_partner_daily_access($role);

const PARTNER_DAILY_LIMIT = 100;

function fetch_all_projects(mysqli $conn): array
{
  $res = $conn->query("SELECT id, name FROM projects ORDER BY name");
  return $res ? $res->fetch_all(MYSQLI_ASSOC) : [];
}

function ensure_partner_daily_access($role = null): void
{
  $role = $role ?? current_user_role();
  if (!can_manage_partner_daily($role)) {
    http_response_code(403);
    exit('Forbidden');
  }
}

function can_delete_partner_daily($role): bool
{
  $role = norm_role($role);
  return in_array($role, ['Administrator', 'Super Administrator'], true);
}

function h($s)
{
  return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

function parse_date($str)
{
  $str = trim((string)$str);
  if ($str === '') {
    return null;
  }
  $ts = strtotime($str);
  return $ts !== false ? date('Y-m-d', $ts) : null;
}

function is_create_submission(array $server, array $request): bool
{
  return (($server['REQUEST_METHOD'] ?? '') === 'POST') && (($request['__action'] ?? '') === 'create');
}

function collect_form_values(array $source): array
{
  $fields = [
    'service_date',
    'driver_partner_id',
    'driver_uuid',
    'partner_id',
    'partner_name',
    'orders_count',
    'amount_total',
  ];
  $out = array_fill_keys($fields, '');
  foreach ($fields as $field) {
    if (array_key_exists($field, $source)) {
      $out[$field] = (string)$source[$field];
    }
  }
  return $out;
}

function upsert_partner_daily_row(
  mysqli $conn,
  string $serviceDate,
  int $driverId,
  string $driverUuid,
  string $driverName,
  ?int $partnerId,
  string $partnerName,
  int $orders,
  float $amount
): bool {
  $existing = $conn->prepare("SELECT id FROM partner_daily WHERE driver_id=? AND service_date=? LIMIT 1");
  if (!$existing) return false;
  $existing->bind_param('is', $driverId, $serviceDate);
  $existing->execute();
  $row = $existing->get_result()->fetch_assoc();
  $partnerIdParam = $partnerId !== null ? (string)$partnerId : null;
  $partnerNameParam = $partnerName !== '' ? $partnerName : null;

  if ($row) {
    $upd = $conn->prepare("
      UPDATE partner_daily
      SET driver_uuid=?, driver_name=?, partner_id=?, partner_name=?, orders_count=?, amount_total=?, status='pending'
      WHERE id=?
    ");
    if (!$upd) return false;
    $id = (int)$row['id'];
    $upd->bind_param(
      'ssissdi',
      $driverUuid,
      $driverName,
      $partnerIdParam,
      $partnerNameParam,
      $orders,
      $amount,
      $id
    );
    return $upd->execute();
  }

  $sql = "INSERT INTO partner_daily
          (service_date, driver_id, driver_uuid, driver_name, partner_id, partner_name, orders_count, amount_total, status)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')";
  $stmt = $conn->prepare($sql);
  if (!$stmt) {
    return false;
  }
  $stmt->bind_param(
    'sissssid',
    $serviceDate,
    $driverId,
    $driverUuid,
    $driverName,
    $partnerIdParam,
    $partnerNameParam,
    $orders,
    $amount
  );
  return $stmt->execute();
}

function find_driver_by_identifiers(mysqli $conn, ?string $partnerId, ?string $partnerUuid): ?array
{
  $partnerId = trim((string)$partnerId);
  $partnerUuid = trim((string)$partnerUuid);

  if ($partnerId === '' && $partnerUuid === '') {
    return null;
  }

  $clauses = [];
  $params = [];
  $types  = '';
  if ($partnerId !== '') {
    $clauses[] = 'partner_id=?';
    $params[]  = $partnerId;
    $types    .= 's';
  }
  if ($partnerUuid !== '') {
    $clauses[] = 'partner_uuid=?';
    $params[]  = $partnerUuid;
    $types    .= 's';
  }
  $sql = "SELECT id, partner_id, partner_uuid, project_id, CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,'')) AS driver_name
          FROM drivers
          WHERE " . implode(' OR ', $clauses) . "
          LIMIT 1";
  $stmt = $conn->prepare($sql);
  if (!$stmt) {
    return null;
  }
  $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  return $row ?: null;
}

function handle_delete_request(mysqli $conn, array $server, array $request, string $role): array
{
  $messages = [];
  if (($server['REQUEST_METHOD'] ?? '') !== 'POST' || ($request['__action'] ?? '') !== 'delete') {
    return $messages;
  }

  if (!can_delete_partner_daily($role)) {
    $messages[] = ['type' => 'danger', 'text' => 'You are not allowed to delete entries.'];
    return $messages;
  }

  $rowId = isset($request['row_id']) ? (int)$request['row_id'] : 0;
  if ($rowId <= 0) {
    $messages[] = ['type' => 'danger', 'text' => 'Invalid entry selected for deletion.'];
    return $messages;
  }

  $stmt = $conn->prepare("DELETE FROM partner_daily WHERE id=? LIMIT 1");
  if (!$stmt) {
    $messages[] = ['type' => 'danger', 'text' => 'Unable to delete entry.'];
    return $messages;
  }
  $stmt->bind_param('i', $rowId);
  $stmt->execute();

  if ($stmt->affected_rows > 0) {
    $messages[] = ['type' => 'success', 'text' => 'Entry deleted.'];
  } else {
    $messages[] = ['type' => 'warning', 'text' => 'Entry not found or already removed.'];
  }
  return $messages;
}

function handle_create_request(mysqli $conn, array $server, array $request): array
{
  $messages = [];
  if (!is_create_submission($server, $request)) {
    return $messages;
  }

  $serviceDate = parse_date($request['service_date'] ?? '');
  $partnerId = ($request['partner_id'] ?? '') !== '' ? (int)$request['partner_id'] : null;
  // Partner name will be resolved from ID or Driver
  $partnerName = '';
  $driverPartnerId = trim($request['driver_partner_id'] ?? '');
  $driverUuidInput = trim($request['driver_uuid'] ?? '');
  $ordersRaw = trim($request['orders_count'] ?? '');
  $amountRaw = trim($request['amount_total'] ?? '');

  $errors = [];
  if (!$serviceDate) {
    $errors[] = 'Date is required.';
  }

  $ordersCount = ctype_digit($ordersRaw) ? (int)$ordersRaw : null;
  if ($ordersCount === null) {
    $errors[] = 'Total orders must be an integer.';
  }

  if ($amountRaw === '' || !preg_match('/^-?\d{1,9}(?:\.\d{1,3})?$/', $amountRaw)) {
    $errors[] = 'Total amount must be a number with up to 3 decimals.';
  }
  $amountFloat = (float)$amountRaw;

  if ($driverPartnerId === '' && $driverUuidInput === '') {
    $errors[] = 'Enter a driver ID or driver UUID.';
  }

  $driverRow = null;
  if (!$errors) {
    $driverRow = find_driver_by_identifiers($conn, $driverPartnerId, $driverUuidInput);
    if (!$driverRow) {
      $errors[] = 'Driver not found for the provided ID/UUID.';
    }
  }

  if (!$errors) {
    $allProjects = fetch_all_projects($conn);
    // If not selected, try driver's project
    if (!$partnerId && !empty($driverRow['project_id'])) {
      $partnerId = (int)$driverRow['project_id'];
    }
    // Resolve name
    if ($partnerId) {
      foreach ($allProjects as $p) {
        if ((int)$p['id'] === $partnerId) {
          $partnerName = $p['name'];
          break;
        }
      }
    }
    if ($partnerName === '') {
      $errors[] = 'Partner is required (select from list or ensure driver is assigned to a project).';
    }
  }

  if ($errors) {
    $messages[] = ['type' => 'danger', 'text' => implode(' ', $errors)];
    return $messages;
  }

  $driverId = (int)$driverRow['id'];
  $driverUuid = trim((string)($driverRow['partner_uuid'] ?? ''));
  if ($driverUuid === '') {
    $driverUuid = trim((string)($driverRow['partner_id'] ?? ''));
  }
  $driverName = trim((string)($driverRow['driver_name'] ?? ''));

  if (upsert_partner_daily_row($conn, $serviceDate, $driverId, $driverUuid, $driverName, $partnerId, $partnerName, $ordersCount, $amountFloat)) {
    $messages[] = ['type' => 'success', 'text' => 'Saved. Marked pending.'];
  } else {
    $messages[] = ['type' => 'danger', 'text' => 'Database error while saving.'];
  }

  return $messages;
}

function build_filter_spec(array $input): array
{
  $from = trim($input['from'] ?? '');
  $to = trim($input['to'] ?? '');
  $partner = trim($input['partner'] ?? '');

  $where = '1=1';
  $params = [];
  $types = '';

  $fromParsed = $from !== '' ? parse_date($from) : null;
  if ($fromParsed) {
    $where .= ' AND pd.service_date >= ?';
    $params[] = $fromParsed;
    $types .= 's';
  }

  $toParsed = $to !== '' ? parse_date($to) : null;
  if ($toParsed) {
    $where .= ' AND pd.service_date <= ?';
    $params[] = $toParsed;
    $types .= 's';
  }

  if ($partner !== '') {
    $where .= ' AND (COALESCE(pd.partner_name, \'\') LIKE ? OR COALESCE(CAST(pd.partner_id AS CHAR), \'\') LIKE ? OR COALESCE(p.name, \'\') LIKE ?)';
    $like = "%$partner%";
    $params[] = $like;
    $params[] = $like;
    $params[] = $like;
    $types .= 'sss';
  }

  return [
    'where' => $where,
    'params' => $params,
    'types' => $types,
    'values' => ['from' => $from, 'to' => $to, 'partner' => $partner],
  ];
}

function fetch_partner_daily(mysqli $conn, array $spec): array
{
  $sql = "SELECT
            pd.id,
            pd.service_date,
            pd.driver_uuid,
            pd.driver_name,
            pd.partner_id,
            pd.partner_name,
            pd.driver_id,
            pd.orders_count,
            pd.amount_total,
            pd.total_orders,
            pd.total_amount,
            pd.status,
            (SELECT s.status FROM submissions s
              WHERE s.driver_id = pd.driver_id
                AND s.service_date = pd.service_date
              ORDER BY s.created_at DESC, s.id DESC
              LIMIT 1) AS submission_status,
            (SELECT s.orders_count FROM submissions s
              WHERE s.driver_id = pd.driver_id
                AND s.service_date = pd.service_date
              ORDER BY s.created_at DESC, s.id DESC
              LIMIT 1) AS sub_orders,
            (SELECT s.cash_deposited FROM submissions s
              WHERE s.driver_id = pd.driver_id
                AND s.service_date = pd.service_date
              ORDER BY s.created_at DESC, s.id DESC
              LIMIT 1) AS sub_amount,
            pd.created_at,
            d.partner_id   AS drv_partner_uuid,
            d.partner_uuid AS drv_uuid_alt,
            CONCAT(COALESCE(d.first_name,''),' ',COALESCE(d.last_name,'')) AS drv_full_name,
            d.project_id   AS drv_project_id,
            p.name         AS project_name
          FROM partner_daily pd
          LEFT JOIN drivers d ON d.id = pd.driver_id
          LEFT JOIN projects p ON p.id = d.project_id
          WHERE {$spec['where']}
          ORDER BY pd.service_date DESC, pd.id DESC
          LIMIT " . PARTNER_DAILY_LIMIT;
  $stmt = $conn->prepare($sql);
  if (!$stmt) {
    return [];
  }
  if ($spec['types']) {
    $stmt->bind_param($spec['types'], ...$spec['params']);
  }
  $stmt->execute();
  $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

  foreach ($rows as &$row) {
    $driverUUID = trim((string)($row['drv_uuid_alt'] ?? ''));
    if ($driverUUID === '') {
      $driverUUID = trim((string)($row['driver_uuid'] ?? ''));
    }
    if ($driverUUID === '' && !empty($row['drv_partner_uuid'])) {
      $driverUUID = trim((string)$row['drv_partner_uuid']);
    }
    $row['driver_uuid_display'] = $driverUUID;

    if (trim((string)($row['driver_name'] ?? '')) === '' && !empty($row['drv_full_name'])) {
      $row['driver_name'] = trim(preg_replace('/\s+/', ' ', $row['drv_full_name']));
    }
    if (($row['partner_name'] ?? '') === '' && !empty($row['project_name'])) {
      $row['partner_name'] = $row['project_name'];
    }

    $ordersCount = isset($row['orders_count']) ? (int)$row['orders_count'] : 0;
    $totalOrders = isset($row['total_orders']) ? (int)$row['total_orders'] : 0;
    $row['orders_display'] = $ordersCount !== 0 ? $ordersCount : $totalOrders;

    $amountPrimary = trim((string)($row['amount_total'] ?? ''));
    $amountSecondary = trim((string)($row['total_amount'] ?? ''));
    $amountVal = $amountPrimary !== '' && (float)$amountPrimary != 0.0
      ? (float)$amountPrimary
      : ((float)$amountSecondary);
    $row['amount_display'] = $amountVal;

    $submissionStatus = trim((string)($row['submission_status'] ?? ''));
    if ($submissionStatus !== '') {
      $row['status_display'] = strtolower($submissionStatus);
    } else {
      $row['status_display'] = strtolower(trim((string)($row['status'] ?? 'pending')));
    }

    if ($row['status_display'] === 'approved') {
      if (isset($row['sub_orders'])) $row['orders_display'] = (int)$row['sub_orders'];
      if (isset($row['sub_amount'])) $row['amount_display'] = (float)$row['sub_amount'];
    }
  }
  unset($row);

  return $rows;
}

function partner_reference(array $row): string
{
  $name = trim((string)($row['partner_name'] ?? ''));
  if ($name === '' && !empty($row['project_name'])) {
    $name = $row['project_name'];
  }
  return $name !== '' ? h($name) : '';
}

function status_badge_class(string $status): string
{
  $status = strtolower($status);
  return match ($status) {
    'approved', 'paid'   => 'success',
    'rejected'           => 'danger',
    'matched'            => 'primary',
    'pending', 'yet_to_pay' => 'warning',
    default              => 'secondary',
  };
}

$messages = array_merge(
  handle_delete_request($conn, $_SERVER, $_POST, $role),
  handle_create_request($conn, $_SERVER, $_POST)
);
$formValues = collect_form_values($_POST);
$filterSpec = build_filter_spec($_GET);
$filterValues = $filterSpec['values'];
$rows = fetch_partner_daily($conn, $filterSpec);
$canDeleteRows = can_delete_partner_daily($role);
$allProjects = fetch_all_projects($conn);

// Calc totals for visible rows
$tot_orders = 0;
$tot_amount = 0.0;
foreach ($rows as $r) {
  $tot_orders += (int)$r['orders_display'];
  $tot_amount += (float)$r['amount_display'];
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Partner Data Entry</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
  <style>
    .stat-card { transition: transform 0.2s, box-shadow 0.2s; }
    .stat-card:hover { transform: translateY(-2px); box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
    .stat-value { font-size: 1.5rem; font-weight: 700; }
    .stat-label { font-size: 0.875rem; font-weight: 600; text-transform: uppercase; letter-spacing: 0.5px; color: #64748b; }
    .filter-card { background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); border: 1px solid #dee2e6; }
  </style>
</head>
<body class="bg-light">
<?php @include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4" style="max-width: 1200px;">
  
  <!-- Header -->
  <div class="d-flex justify-content-between align-items-center mb-4">
    <div>
      <h4 class="mb-1"><i class="bi bi-cloud-arrow-up text-danger"></i> Partner Data Entry</h4>
      <p class="text-muted mb-0 small">Upload and manage daily partner data submissions</p>
    </div>
    <div class="d-flex gap-2">
      <a href="partner_template.php" class="btn btn-outline-secondary btn-sm">
        <i class="bi bi-file-earmark-spreadsheet"></i> CSV Template
      </a>
      <a href="partner_upload.php" class="btn btn-outline-primary btn-sm">
        <i class="bi bi-cloud-upload"></i> Bulk Upload
      </a>
    </div>
  </div>

  <?php foreach ($messages as $m): ?>
    <div class="alert alert-<?php echo h($m['type']); ?> d-flex align-items-center shadow-sm">
      <i class="bi bi-<?php echo $m['type']=='success'?'check-circle':'exclamation-triangle'; ?>-fill me-2"></i>
      <div><?php echo h($m['text']); ?></div>
    </div>
  <?php endforeach; ?>

  <!-- Stats -->
  <div class="row g-3 mb-4">
    <div class="col-md-6">
      <div class="card stat-card border-primary shadow-sm h-100">
        <div class="card-body">
          <div class="d-flex align-items-center mb-2">
            <div class="flex-grow-1"><div class="stat-label text-primary">Total Orders (Visible)</div></div>
            <div class="bg-primary bg-opacity-10 rounded-circle p-3"><i class="bi bi-box-seam text-primary fs-4"></i></div>
          </div>
          <div class="stat-value text-primary"><?php echo number_format($tot_orders); ?></div>
        </div>
      </div>
    </div>
    <div class="col-md-6">
      <div class="card stat-card border-success shadow-sm h-100">
        <div class="card-body">
          <div class="d-flex align-items-center mb-2">
            <div class="flex-grow-1"><div class="stat-label text-success">Total Amount (Visible)</div></div>
            <div class="bg-success bg-opacity-10 rounded-circle p-3"><i class="bi bi-currency-dollar text-success fs-4"></i></div>
          </div>
          <div class="stat-value text-success">OMR <?php echo number_format($tot_amount, 3); ?></div>
        </div>
      </div>
    </div>
  </div>

  <!-- Create Form -->
  <div class="card shadow-sm mb-4">
    <div class="card-header bg-white">
      <h6 class="mb-0"><i class="bi bi-plus-circle"></i> Add New Entry</h6>
    </div>
    <div class="card-body">
      <form method="POST" class="row g-3">
        <input type="hidden" name="__action" value="create">
        <div class="col-md-3">
          <label class="form-label small fw-bold">Date</label>
          <input type="date" name="service_date" class="form-control" required value="<?php echo h($formValues['service_date']); ?>">
        </div>
        <div class="col-md-3">
          <label class="form-label small fw-bold">Driver ID (Partner ID)</label>
          <input type="text" name="driver_partner_id" class="form-control" placeholder="e.g., AK123456" value="<?php echo h($formValues['driver_partner_id']); ?>">
        </div>
        <div class="col-md-3">
          <label class="form-label small fw-bold">Partner UUID</label>
          <input type="text" name="driver_uuid" class="form-control" placeholder="Optional" value="<?php echo h($formValues['driver_uuid']); ?>">
          <div class="form-text small">Or UUID</div>
        </div>
        <div class="col-md-3">
          <label class="form-label small fw-bold">Partner</label>
          <select name="partner_id" class="form-select">
            <option value="">(Select or Auto-detect)</option>
            <?php foreach ($allProjects as $p): ?>
              <option value="<?php echo (int)$p['id']; ?>" <?php echo ($formValues['partner_id'] == $p['id']) ? 'selected' : ''; ?>>
                <?php echo h($p['name']); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-md-3">
          <label class="form-label small fw-bold">Total Orders</label>
          <input type="number" name="orders_count" class="form-control" min="0" step="1" required value="<?php echo h($formValues['orders_count']); ?>">
        </div>
        <div class="col-md-3">
          <label class="form-label small fw-bold">Total Amount (OMR)</label>
          <input type="text" name="amount_total" class="form-control" placeholder="e.g., 52.500" required value="<?php echo h($formValues['amount_total']); ?>">
        </div>
        <div class="col-md-6 d-flex align-items-end justify-content-end">
          <button class="btn btn-danger px-4"><i class="bi bi-save me-1"></i> Save Entry</button>
        </div>
      </form>
    </div>
  </div>

  <!-- Filters & Table -->
  <div class="card shadow-sm">
    <div class="card-header bg-white d-flex justify-content-between align-items-center">
      <h6 class="mb-0"><i class="bi bi-table"></i> Data Records</h6>
      <span class="badge bg-light text-dark border"><?php echo count($rows); ?> visible</span>
    </div>
    <div class="card-body border-bottom bg-light">
      <form class="row g-2 align-items-end">
        <div class="col-auto">
          <label class="form-label small fw-bold mb-1">From Date</label>
          <input type="date" name="from" class="form-control form-control-sm" value="<?php echo h($filterValues['from']); ?>">
        </div>
        <div class="col-auto">
          <label class="form-label small fw-bold mb-1">To Date</label>
          <input type="date" name="to" class="form-control form-control-sm" value="<?php echo h($filterValues['to']); ?>">
        </div>
        <div class="col-auto">
          <label class="form-label small fw-bold mb-1">Partner / Driver</label>
          <input type="text" name="partner" class="form-control form-control-sm" placeholder="Name or ID" value="<?php echo h($filterValues['partner']); ?>">
        </div>
        <div class="col-auto">
          <button class="btn btn-secondary btn-sm"><i class="bi bi-funnel"></i> Filter</button>
          <a href="data_entry.php" class="btn btn-outline-secondary btn-sm" title="Reset"><i class="bi bi-arrow-clockwise"></i></a>
        </div>
      </form>
    </div>

    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th><i class="bi bi-calendar3"></i> Date</th>
            <th><i class="bi bi-person-badge"></i> Driver UUID</th>
            <th><i class="bi bi-person"></i> Driver Name</th>
            <th><i class="bi bi-building"></i> Partner</th>
            <th class="text-end"><i class="bi bi-box-seam"></i> Orders</th>
            <th class="text-end"><i class="bi bi-cash"></i> Amount</th>
            <th>Status</th>
            <th><i class="bi bi-clock"></i> Created</th>
            <?php if ($canDeleteRows): ?>
              <th class="text-end">Actions</th>
            <?php endif; ?>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr>
              <td colspan="<?php echo $canDeleteRows ? 9 : 8; ?>" class="text-center text-muted py-5">
                <i class="bi bi-inbox display-4 opacity-25"></i>
                <div class="mt-2">No records found.</div>
              </td>
            </tr>
          <?php else: foreach ($rows as $r): ?>
            <tr>
              <td class="fw-semibold"><?php echo h($r['service_date']); ?></td>
              <td class="font-monospace small"><?php echo h($r['driver_uuid_display']); ?></td>
              <td><?php echo h($r['driver_name']); ?></td>
              <td><?php echo partner_reference($r); ?></td>
              <td class="text-end fw-bold"><?php echo (int)$r['orders_display']; ?></td>
              <td class="text-end font-monospace"><?php echo number_format((float)$r['amount_display'], 3); ?></td>
              <td>
                <?php $statusDisplay = $r['status_display'] ?? 'pending'; ?>
                <span class="badge bg-<?php echo h(status_badge_class($statusDisplay)); ?>">
                  <?php echo h(ucfirst($statusDisplay)); ?>
                </span>
              </td>
              <td class="small text-muted"><?php echo date('M d H:i', strtotime($r['created_at'])); ?></td>
              <?php if ($canDeleteRows): ?>
                <td class="text-end">
                  <form method="POST" class="d-inline" onsubmit="return confirm('Delete this entry?');">
                    <input type="hidden" name="__action" value="delete">
                    <input type="hidden" name="row_id" value="<?php echo (int)$r['id']; ?>">
                    <button class="btn btn-sm btn-outline-danger" title="Delete"><i class="bi bi-trash"></i></button>
                  </form>
                </td>
              <?php endif; ?>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
