<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

// Protect the page with the asset management policy
if (!can_manage_assets(current_user_role())) {
    http_response_code(403);
    exit('Forbidden');
}

if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$title = 'Bulk Assign Assets via CSV';
$results = null;

// Handle CSV upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['csv_file']['tmp_name'];
        $results = ['success' => 0, 'errors' => []];
        $assigned_by = (int)($_SESSION['user']['id'] ?? 0);

        // For efficiency, pre-fetch drivers and assets into maps
        $drivers_map = $conn->query("SELECT partner_id, id FROM drivers")->fetch_all(MYSQLI_KEY_PAIR);
        $assets_result = $conn->query("SELECT id, name, has_expiry, lifespan_days FROM asset_items WHERE is_active=1");
        $assets_map = [];
        while ($row = $assets_result->fetch_assoc()) {
            $assets_map[$row['name']] = $row;
        }

        $handle = fopen($file, "r");
        $header = fgetcsv($handle); // Skip header row

        if ($header[0] !== 'driver_partner_id' || $header[1] !== 'asset_name' || $header[2] !== 'quantity') {
            $results['errors'][] = "Invalid CSV header. Please use the provided template.";
        } else {
            $stmt = $conn->prepare("INSERT INTO driver_assets (driver_id, asset_item_id, assigned_by_user_id, expiry_date) VALUES (?, ?, ?, ?)");
            $row_num = 1;

            while (($data = fgetcsv($handle)) !== FALSE) {
                $row_num++;
                $driver_id_str = trim($data[0]);
                $asset_name = trim($data[1]);
                $quantity = (int)trim($data[2]);

                // Validation
                if (empty($driver_id_str) || empty($asset_name) || $quantity <= 0) {
                    $results['errors'][] = "Row $row_num: Invalid data. All fields are required and quantity must be positive.";
                    continue;
                }
                if (!isset($drivers_map[$driver_id_str])) {
                    $results['errors'][] = "Row $row_num: Driver with Partner ID '$driver_id_str' not found.";
                    continue;
                }
                if (!isset($assets_map[$asset_name])) {
                    $results['errors'][] = "Row $row_num: Asset with name '$asset_name' not found or is inactive.";
                    continue;
                }

                $driver_id = $drivers_map[$driver_id_str];
                $asset = $assets_map[$asset_name];

                // Process assignment
                for ($i = 0; $i < $quantity; $i++) {
                    $expiry_date = null;
                    if ($asset['has_expiry'] && $asset['lifespan_days']) {
                        $expiry_date = date('Y-m-d', strtotime("+{$asset['lifespan_days']} days"));
                    }
                    $stmt->bind_param('iiis', $driver_id, $asset['id'], $assigned_by, $expiry_date);
                    $stmt->execute();
                    $results['success']++;
                }
            }
            $stmt->close();
        }
        fclose($handle);
    } else {
        $results = ['errors' => ['File upload failed. Please try again.']];
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title><?php echo $title; ?> | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0"><?php echo $title; ?></h5>
        <a href="<?php echo base_url('admin/assign_assets.php'); ?>" class="btn btn-sm btn-outline-secondary">Manual Assignment</a>
    </div>

    <?php if ($results): ?>
        <div class="alert alert-info">
            <h5 class="alert-heading">Upload Summary</h5>
            <p><strong><?php echo $results['success']; ?></strong> total items were successfully assigned.</p>
            <?php if (!empty($results['errors'])): ?>
                <hr>
                <p class="mb-1"><strong>Errors Encountered:</strong></p>
                <ul class="mb-0" style="max-height: 200px; overflow-y: auto;">
                    <?php foreach ($results['errors'] as $error): ?>
                        <li><?php echo e($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <div class="card">
        <div class="card-body">
            <h5 class="card-title">Instructions</h5>
            <ol>
                <li>Download the sample CSV template.</li>
                <li>Fill in the required columns:
                    <ul>
                        <li><code>driver_partner_id</code>: The unique Partner ID of the driver (e.g., TD-123).</li>
                        <li><code>asset_name</code>: The exact name of the asset from the Asset Catalog (e.g., Rider T-Shirt - L).</li>
                        <li><code>quantity</code>: The number of items to assign (e.g., 2).</li>
                    </ul>
                </li>
                <li>Save the file as a CSV.</li>
                <li>Choose the file below and click "Upload and Process".</li>
            </ol>
            <a href="<?php echo base_url('samples/asset_assignment_template.csv'); ?>" class="btn btn-sm btn-success">
                <i class="bi bi-download me-1"></i> Download Sample CSV Template
            </a>
        </div>
    </div>

    <div class="card mt-3">
        <div class="card-body">
            <form method="post" enctype="multipart/form-data">
                <input type="hidden" name="csrf" value="<?php echo $csrf; ?>">
                <div class="mb-3">
                    <label for="csv_file" class="form-label">Upload CSV File</label>
                    <input class="form-control" type="file" id="csv_file" name="csv_file" accept=".csv" required>
                </div>
                <button type="submit" class="btn btn-primary">Upload and Process</button>
            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
