<?php
// public/admin/adjustments_list.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

$role = strtolower(get_current_role() ?? '');
$can_edit = in_array($role, ['super administrator','administrator','finance administrator'], true);
$csrf = get_csrf_token();

$ok=''; $err='';

// ---------------------------------------
// Helpers
// ---------------------------------------
function drivers_options(mysqli $conn, $sel = 0) {
  $h = '';
  $q = $conn->query("SELECT id, partner_id, CONCAT(first_name,' ',last_name) AS name FROM drivers ORDER BY partner_id");
  while ($r = $q->fetch_assoc()) {
    $s = ((int)$sel === (int)$r['id']) ? ' selected' : '';
    $label = trim(($r['partner_id'] ?: '') . ' - ' . ($r['name'] ?: ''));
    $h .= '<option value="'.(int)$r['id'].'"'.$s.'>'.e($label).'</option>';
  }
  return $h;
}
function categories_options(mysqli $conn, $type='', $sel=0) {
  $h = '';
  $stmt = $type==='positive' || $type==='negative'
    ? $conn->prepare("SELECT id, name FROM adjustment_categories WHERE is_active=1 AND type=? ORDER BY name")
    : $conn->prepare("SELECT id, name FROM adjustment_categories WHERE is_active=1 ORDER BY type, name");
  if ($type==='positive' || $type==='negative') $stmt->bind_param('s', $type);
  $stmt->execute();
  $res = $stmt->get_result();
  while ($r = $res->fetch_assoc()) {
    $s = ((int)$sel === (int)$r['id']) ? ' selected' : '';
    $h .= '<option value="'.(int)$r['id'].'"'.$s.'>'.e($r['name']).'</option>';
  }
  return $h;
}

// ---------------------------------------
// CSV Template download
// ---------------------------------------
if (isset($_GET['action']) && $_GET['action']==='template') {
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="driver_adjustments_template.csv"');
  $out = fopen('php://output', 'w');
  fputcsv($out, ['Date','Driver ID','Type','Category','Amount','Remarks']);
  // example rows
  fputcsv($out, ['2025-11-01','TB00001234','positive','Performance Bonus','10.000','October bonus']);
  fputcsv($out, ['2025-11-02','TB00001234','negative','Fuel Usage','3.500','Daily fuel']);
  fclose($out);
  exit;
}

// ---------------------------------------
// Filters - Default to current month
// ---------------------------------------
$f_driver = isset($_GET['driver_id']) && ctype_digit($_GET['driver_id']) ? (int)$_GET['driver_id'] : 0;
$f_type = in_array(($_GET['type'] ?? ''), ['positive','negative'], true) ? $_GET['type'] : '';
$f_cat = isset($_GET['category_id']) && ctype_digit($_GET['category_id']) ? (int)$_GET['category_id'] : 0;

// Default to current month if no date filters provided
$currentMonthStart = date('Y-m-01');
$currentMonthEnd = date('Y-m-t');
$f_from = trim($_GET['from'] ?? '') ?: $currentMonthStart;
$f_to = trim($_GET['to'] ?? '') ?: $currentMonthEnd;

$where=[]; $params=[]; $typestr='';
if ($f_driver>0){ $where[]='a.driver_id=?'; $params[]=$f_driver; $typestr.='i'; }
if ($f_type){ $where[]='a.type=?'; $params[]=$f_type; $typestr.='s'; }
if ($f_cat>0){ $where[]='a.category_id=?'; $params[]=$f_cat; $typestr.='i'; }
if ($f_from && preg_match('/^\d{4}-\d{2}-\d{2}$/',$f_from)){ $where[]='a.adj_date>=?'; $params[]=$f_from; $typestr.='s'; }
if ($f_to && preg_match('/^\d{4}-\d{2}-\d{2}$/',$f_to)){ $where[]='a.adj_date<=?'; $params[]=$f_to; $typestr.='s'; }
$wsql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

// Base query for list/export
$sql_base = "
SELECT a.id, a.driver_id, a.type, a.category_id, a.adj_date, a.amount, a.remarks, a.created_at, a.updated_at,
       d.partner_id, CONCAT(d.first_name,' ',d.last_name) AS driver_name,
       c.name AS category_name
FROM driver_adjustments a
JOIN drivers d ON d.id=a.driver_id
JOIN adjustment_categories c ON c.id=a.category_id
$wsql
ORDER BY a.adj_date DESC, a.id DESC
";

// ---------------------------------------
// CSV Export for current filter
// ---------------------------------------
if (isset($_GET['action']) && $_GET['action']==='export') {
  $stx = $conn->prepare($sql_base);
  if ($typestr) $stx->bind_param($typestr, ...$params);
  $stx->execute();
  $resx = $stx->get_result();

  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="driver_adjustments_export.csv"');
  $out = fopen('php://output', 'w');
  // header
  fputcsv($out, ['Date','Driver ID','Driver Name','Type','Category','Amount','Remarks','Created At']);
  while ($row = $resx->fetch_assoc()) {
    fputcsv($out, [
      $row['adj_date'],
      $row['partner_id'],
      $row['driver_name'],
      $row['type'],
      $row['category_name'],
      number_format((float)$row['amount'], 3, '.', ''),
      $row['remarks'],
      $row['created_at']
    ]);
  }
  fclose($out);
  exit;
}

// ---------------------------------------
// Create / Update single-entry
// ---------------------------------------
if ($can_edit && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf']) && ($_POST['form'] ?? '')==='single') {
  $id = isset($_POST['id']) && ctype_digit($_POST['id']) ? (int)$_POST['id'] : 0;
  $driver_id = isset($_POST['driver_id']) && ctype_digit($_POST['driver_id']) ? (int)$_POST['driver_id'] : 0;
  $type = ($_POST['type'] ?? '') === 'positive' ? 'positive' : 'negative';
  $category_id = isset($_POST['category_id']) && ctype_digit($_POST['category_id']) ? (int)$_POST['category_id'] : 0;
  $adj_date = trim($_POST['adj_date'] ?? '');
  $amount = (string)($_POST['amount'] ?? '0');
  $remarks = trim($_POST['remarks'] ?? '');
  $uid = (int)($_SESSION['user']['id'] ?? 0);

  if ($driver_id<=0 || $category_id<=0 || $adj_date==='') {
    $err = 'Driver, category, and date are required.';
  } elseif (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $adj_date)) {
    $err = 'Date must be YYYY-MM-DD.';
  } elseif (!preg_match('/^-?\d+(\.\d{1,3})?$/', $amount)) {
    $err = 'Amount must be a number with up to 3 decimals.';
  } else {
    if ($id > 0) {
      $st = $conn->prepare("UPDATE driver_adjustments SET driver_id=?, type=?, category_id=?, adj_date=?, amount=?, remarks=?, updated_at=NOW() WHERE id=?");
      $st->bind_param('isssssi', $driver_id, $type, $category_id, $adj_date, $amount, $remarks, $id);
      if ($st->execute()) $ok = 'Adjustment updated.'; else $err='Update failed.';
    } else {
      $st = $conn->prepare("INSERT INTO driver_adjustments(driver_id, type, category_id, adj_date, amount, remarks, created_by) VALUES (?,?,?,?,?,?,?)");
      $st->bind_param('isssssi', $driver_id, $type, $category_id, $adj_date, $amount, $remarks, $uid);
      if ($st->execute()) $ok = 'Adjustment added.'; else $err='Insert failed.';
    }
  }
}

// ---------------------------------------
// Bulk CSV upload
// ---------------------------------------
if ($can_edit && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf']) && ($_POST['form'] ?? '')==='bulk') {
  if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
    $err = 'CSV upload failed.';
  } else {
    $fp = fopen($_FILES['csv']['tmp_name'], 'r');
    if (!$fp) {
      $err = 'Unable to read uploaded file.';
    } else {
      $header = fgetcsv($fp);
      // Accept flexible header but must contain required columns
      $map = ['date'=>-1,'driver id'=>-1,'type'=>-1,'category'=>-1,'amount'=>-1,'remarks'=>-1];
      foreach ($header as $i => $h) {
        $k = strtolower(trim($h));
        if (isset($map[$k])) $map[$k] = $i;
      }
      if (in_array(-1, $map, true)) {
        fclose($fp);
        $err = 'Invalid CSV header. Required: Date, Driver ID, Type, Category, Amount, Remarks';
      } else {
        $added=0; $failed=0; $line=1;
        $uid = (int)($_SESSION['user']['id'] ?? 0);

        // Prepare lookups
        $stmt_driver = $conn->prepare("SELECT id FROM drivers WHERE partner_id=? LIMIT 1");
        $stmt_cat = $conn->prepare("SELECT id FROM adjustment_categories WHERE name=? AND type=? AND is_active=1 LIMIT 1");

        // Insert stmt
        $stmt_ins = $conn->prepare("INSERT INTO driver_adjustments(driver_id, type, category_id, adj_date, amount, remarks, created_by) VALUES (?,?,?,?,?,?,?)");

        while (($row = fgetcsv($fp)) !== false) {
          $line++;
          $date = trim($row[$map['date']] ?? '');
          $driver_pid = trim($row[$map['driver id']] ?? '');
          $type = strtolower(trim($row[$map['type']] ?? ''));
          $category = trim($row[$map['category']] ?? '');
          $amount = trim($row[$map['amount']] ?? '0');
          $remarks = trim($row[$map['remarks']] ?? '');

          // Basic validations
          if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) { $failed++; continue; }
          if (!in_array($type, ['positive','negative'], true)) { $failed++; continue; }
          if (!preg_match('/^-?\d+(\.\d{1,3})?$/', $amount)) { $failed++; continue; }

          // Resolve driver
          $stmt_driver->bind_param('s', $driver_pid);
          $stmt_driver->execute();
          $dr = $stmt_driver->get_result()->fetch_assoc();
          if (!$dr) { $failed++; continue; }
          $driver_id = (int)$dr['id'];

          // Resolve category by name + type
          $stmt_cat->bind_param('ss', $category, $type);
          $stmt_cat->execute();
          $cr = $stmt_cat->get_result()->fetch_assoc();
          if (!$cr) { $failed++; continue; }
          $category_id = (int)$cr['id'];

          // Insert
          $stmt_ins->bind_param('isssssi', $driver_id, $type, $category_id, $date, $amount, $remarks, $uid);
          if ($stmt_ins->execute()) $added++; else $failed++;
        }
        fclose($fp);
        $ok = "Bulk upload complete. Added: {$added}, Skipped: {$failed}.";
      }
    }
  }
}

// ---------------------------------------
// Fetch list for view
// ---------------------------------------
$st = $conn->prepare($sql_base);
if ($typestr) $st->bind_param($typestr, ...$params);
$st->execute();
$rows = $st->get_result()->fetch_all(MYSQLI_ASSOC);

// Totals
$tot_pos = 0.0; $tot_neg = 0.0;
foreach ($rows as $r) {
  if ($r['type']==='positive') $tot_pos += (float)$r['amount'];
  else $tot_neg += (float)$r['amount'];
}
$net = $tot_pos - $tot_neg;
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Driver Adjustments | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
  <style>
    .stat-card {
      transition: transform 0.2s, box-shadow 0.2s;
    }
    .stat-card:hover {
      transform: translateY(-2px);
      box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    }
    .stat-value {
      font-size: 1.5rem;
      font-weight: 700;
    }
    .stat-label {
      font-size: 0.875rem;
      font-weight: 600;
      text-transform: uppercase;
      letter-spacing: 0.5px;
      color: #64748b;
    }
  </style>
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4" style="max-width: 1400px;">
  <div class="d-flex justify-content-between align-items-center mb-4">
    <div>
      <h4 class="mb-1"><i class="bi bi-calculator text-danger"></i> Driver Adjustments</h4>
      <p class="text-muted mb-0 small">Manage bonuses, penalties, and other financial adjustments</p>
    </div>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-secondary btn-sm" href="?action=template">
        <i class="bi bi-file-earmark-spreadsheet"></i> CSV Template
      </a>
      <a class="btn btn-outline-info btn-sm" href="?action=export&<?php echo http_build_query($_GET); ?>">
        <i class="bi bi-download"></i> Export CSV
      </a>
      <?php if ($can_edit): ?>
        <button class="btn btn-danger btn-sm" data-bs-toggle="modal" data-bs-target="#adj_new">
          <i class="bi bi-plus-circle"></i> Add Adjustment
        </button>
        <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#bulk_modal">
          <i class="bi bi-cloud-upload"></i> Bulk Upload
        </button>
      <?php endif; ?>
    </div>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success d-flex align-items-center">
      <i class="bi bi-check-circle-fill me-2"></i>
      <div><?php echo e($ok); ?></div>
    </div>
  <?php endif; ?>
  <?php if ($err): ?>
    <div class="alert alert-danger d-flex align-items-center">
      <i class="bi bi-exclamation-triangle-fill me-2"></i>
      <div><?php echo e($err); ?></div>
    </div>
  <?php endif; ?>

  <!-- Filters -->
  <div class="card shadow-sm mb-4">
    <div class="card-header bg-white">
      <h6 class="mb-0"><i class="bi bi-funnel"></i> Filters</h6>
    </div>
    <div class="card-body">
      <form method="get" class="row g-3">
        <div class="col-md-3">
          <label class="form-label fw-semibold">Driver</label>
          <select name="driver_id" class="form-select">
            <option value="0">All Drivers</option>
            <?php echo drivers_options($conn, $f_driver); ?>
          </select>
        </div>
        <div class="col-md-2">
          <label class="form-label fw-semibold">Type</label>
          <select name="type" class="form-select">
            <option value="">All Types</option>
            <option value="positive" <?php echo $f_type==='positive'?'selected':''; ?>>✓ Positive</option>
            <option value="negative" <?php echo $f_type==='negative'?'selected':''; ?>>✗ Negative</option>
          </select>
        </div>
        <div class="col-md-3">
          <label class="form-label fw-semibold">Category</label>
          <select name="category_id" class="form-select">
            <option value="0">All Categories</option>
            <?php echo categories_options($conn, $f_type, $f_cat); ?>
          </select>
        </div>
        <div class="col-md-2">
          <label class="form-label fw-semibold">From Date</label>
          <input type="date" name="from" class="form-control" value="<?php echo e($f_from); ?>">
        </div>
        <div class="col-md-2">
          <label class="form-label fw-semibold">To Date</label>
          <input type="date" name="to" class="form-control" value="<?php echo e($f_to); ?>">
        </div>
        <div class="col-12 d-flex justify-content-between align-items-center">
          <small class="text-muted">
            <i class="bi bi-info-circle"></i> Showing adjustments for: <strong><?php echo date('F Y', strtotime($f_from)); ?></strong>
          </small>
          <div class="d-flex gap-2">
            <a href="?" class="btn btn-outline-secondary">
              <i class="bi bi-arrow-clockwise"></i> Reset
            </a>
            <button type="submit" class="btn btn-danger">
              <i class="bi bi-search"></i> Apply Filters
            </button>
          </div>
        </div>
      </form>
    </div>
  </div>

  <!-- Totals -->
  <div class="row g-3 mb-4">
    <div class="col-md-4">
      <div class="card stat-card border-success shadow-sm">
        <div class="card-body">
          <div class="d-flex align-items-center mb-2">
            <div class="flex-grow-1">
              <div class="stat-label text-success">Total Positives</div>
            </div>
            <div class="bg-success bg-opacity-10 rounded-circle p-3">
              <i class="bi bi-plus-circle-fill text-success fs-4"></i>
            </div>
          </div>
          <div class="stat-value text-success">OMR <?php echo number_format($tot_pos, 3); ?></div>
          <div class="small text-muted mt-1">Bonuses and incentives</div>
        </div>
      </div>
    </div>
    <div class="col-md-4">
      <div class="card stat-card border-danger shadow-sm">
        <div class="card-body">
          <div class="d-flex align-items-center mb-2">
            <div class="flex-grow-1">
              <div class="stat-label text-danger">Total Negatives</div>
            </div>
            <div class="bg-danger bg-opacity-10 rounded-circle p-3">
              <i class="bi bi-dash-circle-fill text-danger fs-4"></i>
            </div>
          </div>
          <div class="stat-value text-danger">OMR <?php echo number_format($tot_neg, 3); ?></div>
          <div class="small text-muted mt-1">Fuel, fines, penalties</div>
        </div>
      </div>
    </div>
    <div class="col-md-4">
      <div class="card stat-card border-<?php echo ($net>=0)?'success':'danger'; ?> shadow-sm">
        <div class="card-body">
          <div class="d-flex align-items-center mb-2">
            <div class="flex-grow-1">
              <div class="stat-label">Net Total</div>
            </div>
            <div class="bg-<?php echo ($net>=0)?'success':'danger'; ?> bg-opacity-10 rounded-circle p-3">
              <i class="bi bi-calculator-fill text-<?php echo ($net>=0)?'success':'danger'; ?> fs-4"></i>
            </div>
          </div>
          <div class="stat-value text-<?php echo ($net>=0)?'success':'danger'; ?>">OMR <?php echo number_format($net, 3); ?></div>
          <div class="small text-muted mt-1">Positives minus negatives</div>
        </div>
      </div>
    </div>
  </div>

  <!-- Table -->
  <div class="card shadow-sm">
    <div class="card-header bg-white">
      <h6 class="mb-0"><i class="bi bi-table"></i> Adjustment Records (<?php echo count($rows); ?>)</h6>
    </div>
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table table-hover align-middle mb-0">
          <thead class="table-light">
            <tr>
              <th style="width:110px;"><i class="bi bi-calendar3"></i> Date</th>
              <th><i class="bi bi-person"></i> Driver</th>
              <th style="width:100px;"><i class="bi bi-tag"></i> Type</th>
              <th><i class="bi bi-folder"></i> Category</th>
              <th class="text-end" style="width:140px;"><i class="bi bi-cash"></i> Amount (OMR)</th>
              <th><i class="bi bi-chat-left-text"></i> Remarks</th>
              <th style="width:140px;"><i class="bi bi-clock"></i> Created</th>
              <th class="text-end" style="width:90px;">Action</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$rows): ?>
              <tr>
                <td colspan="8" class="text-center text-muted py-5">
                  <i class="bi bi-inbox" style="font-size: 48px; opacity: 0.3;"></i>
                  <div class="mt-2">No adjustments found for the selected filters</div>
                </td>
              </tr>
            <?php else: foreach ($rows as $r): ?>
              <tr>
                <td class="fw-semibold"><?php echo e($r['adj_date']); ?></td>
                <td>
                  <div class="fw-semibold"><?php echo e($r['partner_id']); ?></div>
                  <div class="text-muted small"><?php echo e($r['driver_name']); ?></div>
                </td>
                <td>
                  <span class="badge <?php echo $r['type']==='positive'?'text-bg-success':'text-bg-danger'; ?>">
                    <?php echo $r['type']==='positive'?'<i class="bi bi-plus-circle"></i>':'<i class="bi bi-dash-circle"></i>'; ?>
                    <?php echo e(ucfirst($r['type'])); ?>
                  </span>
                </td>
                <td><?php echo e($r['category_name']); ?></td>
                <td class="text-end fw-semibold"><?php echo number_format((float)$r['amount'], 3); ?></td>
                <td class="small"><?php echo e($r['remarks']); ?></td>
                <td><small class="text-muted"><?php echo date('M d, Y H:i', strtotime($r['created_at'])); ?></small></td>
                <td class="text-end">
                  <?php if ($can_edit): ?>
                  <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#edit_<?php echo (int)$r['id']; ?>">
                    <i class="bi bi-pencil"></i>
                  </button>
                  <?php else: ?>
                  <span class="text-muted small">View only</span>
                  <?php endif; ?>
                </td>
              </tr>

              <?php if ($can_edit): ?>
              <!-- Edit modal -->
              <div class="modal fade" id="edit_<?php echo (int)$r['id']; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog">
                  <div class="modal-content">
                    <form method="post">
                      <?php echo csrf_input(); ?>
                      <div class="modal-header">
                        <h6 class="modal-title">Edit Adjustment</h6>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                      </div>
                      <div class="modal-body">
                        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
                        <input type="hidden" name="form" value="single">
                        <input type="hidden" name="id" value="<?php echo (int)$r['id']; ?>">
                        <div class="mb-2">
                          <label class="form-label">Driver</label>
                          <select name="driver_id" class="form-select" required>
                            <?php echo drivers_options($conn, (int)$r['driver_id']); ?>
                          </select>
                        </div>
                        <div class="mb-2">
                          <label class="form-label">Type</label>
                          <select name="type" class="form-select" required>
                            <option value="positive" <?php echo $r['type']==='positive'?'selected':''; ?>>Positive</option>
                            <option value="negative" <?php echo $r['type']==='negative'?'selected':''; ?>>Negative</option>
                          </select>
                        </div>
                        <div class="mb-2">
                          <label class="form-label">Category</label>
                          <select name="category_id" class="form-select" required>
                            <?php echo categories_options($conn, $r['type'], (int)$r['category_id']); ?>
                          </select>
                        </div>
                        <div class="mb-2">
                          <label class="form-label">Date</label>
                          <input type="date" name="adj_date" class="form-control" value="<?php echo e($r['adj_date']); ?>" required>
                        </div>
                        <div class="mb-2">
                          <label class="form-label">Amount (OMR)</label>
                          <input type="text" name="amount" class="form-control" inputmode="decimal" value="<?php echo e($r['amount']); ?>" placeholder="0.000" required>
                        </div>
                        <div>
                          <label class="form-label">Remarks</label>
                          <input type="text" name="remarks" class="form-control" value="<?php echo e($r['remarks']); ?>">
                        </div>
                      </div>
                      <div class="modal-footer">
                        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                        <button class="btn btn-danger">Save</button>
                      </div>
                    </form>
                  </div>
                </div>
              </div>
              <?php endif; ?>

            <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<?php if ($can_edit): ?>
<!-- New Adjustment Modal -->
<div class="modal fade" id="adj_new" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <form method="post">
        <?php echo csrf_input(); ?>
        <div class="modal-header">
          <h6 class="modal-title">Add Adjustment</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
          <input type="hidden" name="form" value="single">
          <div class="mb-2">
            <label class="form-label">Driver</label>
            <select name="driver_id" class="form-select" required>
              <?php echo drivers_options($conn, 0); ?>
            </select>
          </div>
          <div class="mb-2">
            <label class="form-label">Type</label>
            <select name="type" class="form-select" required>
              <option value="positive">Positive</option>
              <option value="negative">Negative</option>
            </select>
          </div>
          <div class="mb-2">
            <label class="form-label">Category</label>
            <select name="category_id" class="form-select" required>
              <?php echo categories_options($conn, '', 0); ?>
            </select>
          </div>
          <div class="mb-2">
            <label class="form-label">Date</label>
            <input type="date" name="adj_date" class="form-control" required>
          </div>
          <div class="mb-2">
            <label class="form-label">Amount (OMR)</label>
            <input type="text" name="amount" class="form-control" inputmode="decimal" placeholder="0.000" required>
          </div>
          <div>
            <label class="form-label">Remarks</label>
            <input type="text" name="remarks" class="form-control" placeholder="Optional">
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-danger">Add</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Bulk Upload Modal -->
<div class="modal fade" id="bulk_modal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <form method="post" enctype="multipart/form-data">
        <div class="modal-header">
          <h6 class="modal-title">Bulk Upload Adjustments (CSV)</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
          <input type="hidden" name="form" value="bulk">
          <div class="mb-2">
            <div class="small text-muted mb-1">Required columns in this order:</div>
            <div class="border rounded p-2 small bg-light">Date, Driver ID, Type, Category, Amount, Remarks</div>
          </div>
          <div class="mb-2">
            <label class="form-label">CSV File</label>
            <input type="file" name="csv" class="form-control" accept=".csv" required>
          </div>
          <div class="small text-muted">
            Type must be positive or negative. Date must be YYYY-MM-DD.
            Category must exist and match the type. Driver ID must be a valid partner ID.
          </div>
        </div>
        <div class="modal-footer">
          <a class="btn btn-outline-secondary" href="?action=template">Download Template</a>
          <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-danger">Upload</button>
        </div>
      </form>
    </div>
  </div>
</div>
<?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
