<?php
/**
 * uploads.php
 * Shared helpers for file validation and storage paths.
 * All files are stored OUTSIDE public web root in /storage/ directory.
 * - Submissions: /storage/submissions/YYYY/MM/UUID/YYYY-MM-DD/{base}-{ts}.ext
 * - KYD docs:    /storage/docs/UUID/{fixed-base}.ext
 */

declare(strict_types=1);

require_once __DIR__ . '/constants.php';

/* ------------------------------
   Basic validators
--------------------------------*/

function allowed_image(string $tmpPath, string $origName): bool {
  if (!is_file($tmpPath)) return false;

  $f = new finfo(FILEINFO_MIME_TYPE);
  $mime = (string)($f->file($tmpPath) ?: '');

  $okMime = ['image/jpeg','image/png','image/webp','image/gif'];
  if (!in_array($mime, $okMime, true)) return false;

  $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
  $okExt = ['jpg','jpeg','png','webp','gif'];
  return in_array($ext, $okExt, true);
}

function ensure_dir(string $absPath): bool {
  if (is_dir($absPath)) return true;
  return @mkdir($absPath, 0775, true);
}

function guess_image_extension(string $tmpPath, string $origName): string {
  $f = new finfo(FILEINFO_MIME_TYPE);
  $mime = (string)($f->file($tmpPath) ?: '');
  $map = [
    'image/jpeg' => 'jpg',
    'image/png'  => 'png',
    'image/webp' => 'webp',
    'image/gif'  => 'gif',
  ];
  if (isset($map[$mime])) return $map[$mime];
  $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
  return $ext !== '' ? $ext : 'jpg';
}

/* ------------------------------
   Submission storage (daily)
   /storage/submissions/YYYY/MM/UUID/YYYY-MM-DD/
--------------------------------*/

function build_submission_dir(string $driverUuid, string $serviceDate): array {
  // $serviceDate = 'YYYY-MM-DD'
  $dt = DateTime::createFromFormat('Y-m-d', $serviceDate) ?: new DateTime('today');
  $rel = $dt->format('Y') . '/' .
         $dt->format('m') . '/' .
         $driverUuid . '/' .
         $dt->format('Y-m-d');
  $abs = STORAGE_DIR . '/submissions/' . $rel;
  ensure_dir($abs);
  // Return absolute path and storage-relative path for DB storage
  return [$abs, 'storage/submissions/' . $rel];
}

/**
 * Save an uploaded submission file with a time suffix to avoid clashes.
 * Returns the storage path, e.g. storage/submissions/2025/10/TB00001234/2025-10-24/orders-1730054321.jpg
 */
function save_submission_file(array $file, string $absDir, string $relDir, string $base): string {
  if (!isset($file['tmp_name'], $file['name'])) return '';
  if (!allowed_image($file['tmp_name'], $file['name'])) return '';

  $ext = guess_image_extension($file['tmp_name'], $file['name']);
  $safeBase = preg_replace('/[^a-z0-9_]/i', '_', $base);
  $targetAbs = rtrim($absDir, '/') . '/' . $safeBase . '-' . time() . '.' . $ext;
  $targetRel = rtrim($relDir, '/') . '/' . basename($targetAbs);

  if (!move_uploaded_file($file['tmp_name'], $targetAbs)) {
    if (!@copy($file['tmp_name'], $targetAbs)) return '';
  }
  @chmod($targetAbs, 0644);
  return $targetRel;
}

/* ------------------------------
   KYD docs storage (fixed names)
   /storage/docs/UUID/
   id_front.jpg, id_back.jpg, dl_front.jpg, ...
--------------------------------*/

function build_docs_dir(string $driverUuid): array {
  $rel = $driverUuid;
  $abs = STORAGE_DIR . '/docs/' . $rel;
  ensure_dir($abs);
  // Return absolute path and storage-relative path for DB storage
  return [$abs, 'storage/docs/' . $rel];
}

/**
 * Save a KYD document with deterministic filename.
 * $base like: id_front, id_back, dl_front, dl_back, vr_front, vr_back, selfie, good_conduct, medical, other
 * Returns storage path, e.g. storage/docs/TB00001234/id_front.jpg
 * Overwrites existing to keep a single current file per type.
 */
function save_doc_file(array $file, string $absDir, string $relDir, string $base): string {
  if (!isset($file['tmp_name'], $file['name'])) return '';
  if (!allowed_image($file['tmp_name'], $file['name'])) return '';

  $ext = guess_image_extension($file['tmp_name'], $file['name']);
  $safeBase = preg_replace('/[^a-z0-9_]/i', '_', $base);
  $targetAbs = rtrim($absDir, '/') . '/' . $safeBase . '.' . $ext;
  $targetRel = rtrim($relDir, '/') . '/' . basename($targetAbs);

  if (!move_uploaded_file($file['tmp_name'], $targetAbs)) {
    if (!@copy($file['tmp_name'], $targetAbs)) return '';
  }
  @chmod($targetAbs, 0644);
  return $targetRel;
}

/* ------------------------------
   Branding storage
   /storage/branding/{type}.ext
   admin_logo, admin_login_logo, driver_logo, driver_login_logo
--------------------------------*/

function save_branding_file(array $file, string $type): string {
  $allowed_types = ['admin_logo', 'admin_login_logo', 'driver_logo', 'driver_login_logo'];
  if (!in_array($type, $allowed_types, true)) return '';

  if (!isset($file['tmp_name'], $file['name'])) return '';
  if ($file['error'] !== UPLOAD_ERR_OK) return '';
  if (!allowed_image($file['tmp_name'], $file['name'])) return '';

  $ext = guess_image_extension($file['tmp_name'], $file['name']);
  $abs_dir = STORAGE_DIR . '/branding';
  ensure_dir($abs_dir);

  // Remove old file if exists (may have different extension)
  foreach (glob($abs_dir . '/' . $type . '.*') as $old) {
    @unlink($old);
  }

  $target = $abs_dir . '/' . $type . '.' . $ext;
  if (!move_uploaded_file($file['tmp_name'], $target)) {
    if (!@copy($file['tmp_name'], $target)) return '';
  }
  @chmod($target, 0644);

  return 'storage/branding/' . $type . '.' . $ext;
}

function delete_branding_file(string $type): bool {
  $allowed_types = ['admin_logo', 'admin_login_logo', 'driver_logo', 'driver_login_logo'];
  if (!in_array($type, $allowed_types, true)) return false;

  $abs_dir = STORAGE_DIR . '/branding';
  foreach (glob($abs_dir . '/' . $type . '.*') as $old) {
    @unlink($old);
  }
  return true;
}
