<?php
// includes/settings.php
// Helper functions to fetch application-wide settings.

if (!function_exists('app_setting')) {
    /**
     * Fetch a single setting, falling back to $default when missing.
     *
     * @param string      $key
     * @param mixed       $default
     * @param mysqli|null $conn Optional connection to use for loading (helpful before global $conn exists).
     */
    function app_setting(string $key, $default = null, ?mysqli $conn = null) {
        $settings = app_settings_cache($conn);
        return array_key_exists($key, $settings) ? $settings[$key] : $default;
    }
}

if (!function_exists('app_settings_cache')) {
    /**
     * Return cached associative array of settings for this request.
     * Cache can be invalidated by calling invalidate_settings_cache().
     */
    function app_settings_cache(?mysqli $conn = null): array {
        static $cache = null;

        // Check if cache was invalidated
        if (!empty($GLOBALS['__settings_cache_invalid'])) {
            $cache = null;
            unset($GLOBALS['__settings_cache_invalid']);
        }

        if ($cache !== null) {
            return $cache;
        }
        $cache = [];

        if (!($conn instanceof mysqli) && isset($GLOBALS['conn']) && $GLOBALS['conn'] instanceof mysqli) {
            $conn = $GLOBALS['conn'];
        }

        if (!($conn instanceof mysqli)) {
            return $cache;
        }

        try {
            $res = $conn->query("SHOW TABLES LIKE 'app_settings'");
            if (!$res || $res->num_rows === 0) {
                return $cache;
            }
        } catch (Throwable $e) {
            return $cache;
        }

        try {
            $result = $conn->query("SELECT setting_key, setting_value FROM app_settings");
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $cache[$row['setting_key']] = $row['setting_value'];
                }
                $result->free();
            }
        } catch (Throwable $e) {
            // swallow and keep cache empty
        }

        return $cache;
    }
}

if (!function_exists('invalidate_settings_cache')) {
    /**
     * Clear the settings cache so changes take effect immediately within this request.
     */
    function invalidate_settings_cache(): void {
        $GLOBALS['__settings_cache_invalid'] = true;
    }
}
