<?php
// ============================================================================
// helpers.php - Clean production version for Hisbuu
// ============================================================================

// Escape helper
if (!function_exists('e')) {
    function e($v) {
        return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
    }
}

// Get the base URL path (e.g., /hisbuu-staging/public)
if (!function_exists('get_base_path')) {
    function get_base_path(): string {
        static $basePath = null;
        if ($basePath !== null) {
            return $basePath;
        }

        // Get the script path and extract the base directory
        $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
        $requestUri = $_SERVER['REQUEST_URI'] ?? '';

        // Extract the path up to /public
        if (preg_match('#^(.*?/public)#', $scriptName, $matches)) {
            $basePath = $matches[1];
        } elseif (preg_match('#^(.*?/public)#', $requestUri, $matches)) {
            $basePath = $matches[1];
        } else {
            $basePath = '/public';
        }

        return $basePath;
    }
}

// Get the base URL for the application (e.g., /hisbuu-staging/public)
if (!function_exists('base_url')) {
    function base_url(string $path = ''): string {
        $base = get_base_path();
        $path = ltrim($path, '/');
        return $base . ($path !== '' ? '/' . $path : '');
    }
}

// Build a web URL for stored files
if (!function_exists('file_url')) {
    function file_url(string $p): string {
        $p = trim($p);
        if ($p === '') return '';

        // Full URL already
        if (preg_match('~^https?://~i', $p)) {
            return $p;
        }

        $base = get_base_path();

        // Storage paths - route through serve.php
        // Format: /storage/submissions/... or /storage/docs/... or /storage/branding/...
        if (preg_match('~^/?storage/(submissions|docs|branding)/(.+)$~', $p, $m)) {
            return $base . '/serve.php?type=' . urlencode($m[1]) . '&path=' . urlencode($m[2]);
        }

        // Absolute path under /public
        if ($p[0] === '/') {
            if (stripos($p, '/public/') === 0) return $p;
            if (stripos($p, '/uploads/') === 0) return $base . $p;
            return $base . $p;
        }

        // Storage paths without leading slash
        if (preg_match('~^storage/(submissions|docs|branding)/(.+)$~', $p, $m)) {
            return $base . '/serve.php?type=' . urlencode($m[1]) . '&path=' . urlencode($m[2]);
        }

        // Relative uploads/...
        if (stripos($p, 'uploads/') === 0) {
            return $base . '/' . $p;
        }

        // Fallback
        return $base . '/uploads/' . ltrim($p, '/');
    }
}

// Redirect helper
if (!function_exists('redirect_to')) {
    function redirect_to(string $url) {
        header('Location: ' . $url);
        exit;
    }
}

// Flash messages
if (!function_exists('flash')) {
    function flash(string $key, ?string $message = null) {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        if ($message === null) {
            if (!isset($_SESSION['flash'][$key])) return null;
            $msg = $_SESSION['flash'][$key];
            unset($_SESSION['flash'][$key]);
            return $msg;
        }

        $_SESSION['flash'][$key] = $message;
        return null;
    }
}

// Format date
if (!function_exists('format_date')) {
    function format_date(?string $dateStr): string {
        if (!$dateStr) return '';
        $ts = strtotime($dateStr);
        if (!$ts) return '';
        return date('Y-m-d H:i', $ts);
    }
}

// Human readable file size
if (!function_exists('human_filesize')) {
    function human_filesize($bytes, $decimals = 2) {
        $size = ['B', 'KB', 'MB', 'GB', 'TB'];
        $factor = floor((strlen($bytes) - 1) / 3);
        return sprintf("%.{$decimals}f", $bytes / pow(1024, $factor)) . ($size[$factor] ?? '');
    }
}

// Session helpers
if (!function_exists('session_get')) {
    function session_get(string $key, $default = null) {
        if (session_status() === PHP_SESSION_NONE) session_start();
        return $_SESSION[$key] ?? $default;
    }
}

if (!function_exists('session_set')) {
    function session_set(string $key, $value): void {
        if (session_status() === PHP_SESSION_NONE) session_start();
        $_SESSION[$key] = $value;
    }
}

if (!function_exists('session_forget')) {
    function session_forget(string $key): void {
        if (session_status() === PHP_SESSION_NONE) session_start();
        unset($_SESSION[$key]);
    }
}

// CSRF helpers
if (!function_exists('csrf_token')) {
    function csrf_token(): string {
        if (session_status() === PHP_SESSION_NONE) session_start();
        if (empty($_SESSION['csrf'])) {
            $_SESSION['csrf'] = bin2hex(random_bytes(16));
        }
        return $_SESSION['csrf'];
    }
}

if (!function_exists('csrf_input')) {
    function csrf_input(): string {
        $token = csrf_token();
        return '<input type="hidden" name="csrf" value="' . e($token) . '">';
    }
}

// App info helper
if (!function_exists('app_info')) {
    function app_info(): array {
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $isLocal = stripos($host, 'localhost') !== false || stripos($host, '127.0.0.1') !== false;
        return [
            'app'  => 'Hisbuu',
            'env'  => $isLocal ? 'local' : 'production',
            'base' => get_base_path()
        ];
    }
}
