# Bootstrap Files Guide - Eliminating Code Duplication

## Overview

Two bootstrap files have been created to eliminate the repeated require statements and common initialization code across the application:

- [`includes/admin_bootstrap.php`](includes/admin_bootstrap.php) - For all admin pages
- [`includes/driver_bootstrap.php`](includes/driver_bootstrap.php) - For all driver pages

---

## What These Files Do

Each bootstrap file automatically:

1. **Configures the session** (must be first)
2. **Starts the session** if not already started
3. **Loads authentication** (admin or driver)
4. **Loads the policy system** (admin only)
5. **Loads helper functions**
6. **Loads the settings system**
7. **Connects to the database**
8. **Ensures user is logged in**
9. **Initializes CSRF token**
10. **Provides convenience functions**

---

## Usage Examples

### Before (Old Way - 5+ lines of requires)

```php
<?php
require_once __DIR__ . '/../../includes/auth_check.php';
require_once __DIR__ . '/../../includes/policy.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/settings.php';
require_once __DIR__ . '/../../config/db.php';

require_login();

if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$role = current_user_role();
```

### After (New Way - 1 line)

```php
<?php
require_once __DIR__ . '/../includes/admin_bootstrap.php';

$role = get_current_role();
$csrf = get_csrf_token();
```

---

## Admin Pages Usage

### Basic Admin Page

```php
<?php
// Single include replaces 5+ requires
require_once __DIR__ . '/../includes/admin_bootstrap.php';

$role = get_current_role();

// Check permissions
if (!can('can_review_submissions')) {
    http_response_code(403);
    exit('Forbidden');
}

// Handle POST with CSRF validation
if (is_valid_post()) {
    // Your form processing logic here
    $name = trim($_POST['name'] ?? '');
    // ...
}
?>
<!doctype html>
<html>
<head>
    <title>My Admin Page</title>
</head>
<body>
    <?php include __DIR__ . '/../includes/admin_nav.php'; ?>
    
    <form method="post">
        <input type="hidden" name="csrf" value="<?php echo get_csrf_token(); ?>">
        <!-- Your form fields -->
    </form>
</body>
</html>
```

---

## Driver Pages Usage

### Basic Driver Page

```php
<?php
// Single include replaces 4+ requires
require_once __DIR__ . '/../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$partner_id = get_current_partner_id();

// Handle POST with CSRF validation
if (is_valid_post()) {
    // Your form processing logic here
}
?>
<!doctype html>
<html>
<head>
    <title>My Driver Page</title>
</head>
<body>
    <?php include __DIR__ . '/../includes/driver_ui_start.php'; ?>
    
    <form method="post">
        <input type="hidden" name="csrf" value="<?php echo get_csrf_token(); ?>">
        <!-- Your form fields -->
    </form>
    
    <?php include __DIR__ . '/../includes/driver_ui_end.php'; ?>
</body>
</html>
```

---

## Available Helper Functions

### Both Admin & Driver Bootstraps

| Function | Description |
|----------|-------------|
| `get_csrf_token()` | Get the current CSRF token |
| `validate_csrf_token(?string $token)` | Validate a CSRF token |
| `is_valid_post()` | Check if POST with valid CSRF |

### Admin Bootstrap Only

| Function | Description |
|----------|-------------|
| `get_current_role()` | Get current user's role |
| `can(callable $check)` | Check if user has permission |

### Driver Bootstrap Only

| Function | Description |
|----------|-------------|
| `get_current_driver_id()` | Get current driver's ID |
| `get_current_partner_id()` | Get current driver's partner ID |

---

## Migration Examples

### Example 1: cities_manage.php

**Before:**
```php
<?php
require_once __DIR__ . '/../../includes/auth_check.php';
require_once __DIR__ . '/../../includes/policy.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../config/db.php';

require_login();

$role = current_user_role();
$can_manage = can_manage_lookups($role);

// CSRF
if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];
```

**After:**
```php
<?php
require_once __DIR__ . '/../includes/admin_bootstrap.php';

$can_manage = can('can_manage_lookups');
$csrf = get_csrf_token();
```

### Example 2: driver/login.php

**Before:**
```php
<?php
require_once __DIR__ . '/../../includes/driver_auth.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/settings.php';
require_once __DIR__ . '/../../config/db.php';

$err = '';
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

redirect_driver_if_logged_in();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    // Login logic
}
```

**After:**
```php
<?php
// Note: login page is special case - don't use bootstrap that requires login
// Instead use minimal includes:
require_once __DIR__ . '/../../includes/driver_auth.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/settings.php';
require_once __DIR__ . '/../../includes/config/db.php';

$err = '';
$csrf = get_csrf_token(); // Use helper if available

redirect_driver_if_logged_in();

if (is_valid_post()) {
    // Login logic
}
```

---

## Files to Migrate

### Admin Files (50+ files)

Replace the require block in these files:
- `public/admin/adjustments_list.php`
- `public/admin/adjustment_categories_manage.php`
- `public/admin/ajax_zones_by_city.php`
- `public/admin/approvals_store.php`
- `public/admin/assets.php`
- `public/admin/assign_assets.php`
- `public/admin/assign_assets_bulk.php`
- `public/admin/bank_upload.php`
- `public/admin/cities_manage.php`
- `public/admin/data_entry.php`
- `public/admin/data_entry_store.php`
- `public/admin/driver_create.php`
- `public/admin/driver_doc_sets_review.php`
- `public/admin/driver_edit.php`
- `public/admin/driver_list.php`
- `public/admin/driver_view.php`
- `public/admin/earnings_monthly_upload.php`
- `public/admin/lookups.php`
- `public/admin/nationalities_manage.php`
- `public/admin/partner_template.php`
- `public/admin/partner_upload.php`
- `public/admin/pending_approvals.php`
- `public/admin/payouts/payouts.php`
- `public/admin/payouts/payout_cycle_view.php`
- `public/admin/payouts/payout_driver_view.php`
- `public/admin/payouts/payout_recalculate.php`
- `public/admin/preview.php`
- `public/admin/projects_manage.php`
- `public/admin/reconcile_edit.php`
- `public/admin/reconcile_list.php`
- `public/admin/reconcile_store.php`
- `public/admin/salary_schemes_manage.php`
- `public/admin/settings.php`
- `public/admin/submissions_gaps.php`
- `public/admin/submissions_list.php`
- `public/admin/submissions_pending.php`
- `public/admin/submissions_summary.php`
- `public/admin/submission_action.php`
- `public/admin/types_manage.php`
- `public/admin/users_manage.php`
- `public/admin/users_new.php`
- `public/admin/users_reset.php`
- `public/admin/users_reset_store.php`
- `public/admin/users_store.php`
- `public/admin/users_toggle.php`
- `public/admin/zones_by_city.php`
- `public/admin/zones_manage.php`

### Driver Files (15+ files)

Replace the require block in these files:
- `public/driver/adjustments.php`
- `public/driver/dashboard.php`
- `public/driver/history.php`
- `public/driver/kyd_manage.php`
- `public/driver/my_gear.php`
- `public/driver/my_submissions.php`
- `public/driver/payout_slip.php`
- `public/driver/payout_view.php`
- `public/driver/payouts_list.php`
- `public/driver/reports.php`
- `public/driver/resubmit.php`
- `public/driver/store_submission.php`
- `public/driver/submit.php`
- `public/driver/uploads.php`
- `public/driver/view.php`

---

## Special Cases

### Login Pages

Login pages should NOT use the bootstrap files because they need to work before the user is logged in. Use minimal includes instead:

**Admin Login:**
```php
require_once __DIR__ . '/../includes/helpers.php';
require_once __DIR__ . '/../includes/auth_check.php';
require_once __DIR__ . '/../includes/settings.php';
require_once __DIR__ . '/../config/db.php';
```

**Driver Login:**
```php
require_once __DIR__ . '/../includes/driver_auth.php';
require_once __DIR__ . '/../includes/helpers.php';
require_once __DIR__ . '/../includes/settings.php';
require_once __DIR__ . '/../config/db.php';
```

---

## Benefits

1. **Reduced Code**: 5-10 lines → 1 line
2. **Consistency**: All pages use same initialization
3. **Maintainability**: Changes to initialization only need to be made in one place
4. **Less Error Prone**: No chance of forgetting a required file
5. **Cleaner Code**: Focus on business logic, not boilerplate
